<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Volunteer Model
 * Handles all volunteer-related database operations
 * Location: application/models/Volunteer_model.php
 */
class Volunteer_model extends CI_Model {

    private $table = 'volunteers';

    public function __construct() {
        parent::__construct();
    }

    /**
     * Get all volunteers with pagination
     */
    public function get_volunteers($limit = 10, $offset = 0, $status = null) {
        $this->db->select('volunteers.*, admin_users.full_name as approved_by_name');
        $this->db->from($this->table);
        $this->db->join('admin_users', 'admin_users.id = volunteers.approved_by', 'left');
        
        if ($status) {
            $this->db->where('volunteers.status', $status);
        }
        
        $this->db->order_by('volunteers.application_date', 'DESC');
        $this->db->limit($limit, $offset);
        
        $query = $this->db->get();
        return $query->result();
    }

    /**
     * Count volunteers by status
     */
    public function count_volunteers($status = null) {
        if ($status) {
            $this->db->where('status', $status);
        }
        return $this->db->count_all_results($this->table);
    }

    /**
     * Get volunteer by ID
     */
    public function get_volunteer($id) {
        $this->db->select('volunteers.*, admin_users.full_name as approved_by_name');
        $this->db->from($this->table);
        $this->db->join('admin_users', 'admin_users.id = volunteers.approved_by', 'left');
        $this->db->where('volunteers.id', $id);
        
        $query = $this->db->get();
        return $query->row();
    }

    /**
     * Get active/approved volunteers for public display
     */
    public function get_active_volunteers($limit = null) {
        $this->db->select('id, full_name, occupation, skills, profile_photo, areas_of_interest');
        $this->db->from($this->table);
        $this->db->where('status', 'active');
        $this->db->order_by('approved_at', 'DESC');
        
        if ($limit) {
            $this->db->limit($limit);
        }
        
        $query = $this->db->get();
        return $query->result();
    }

    /**
     * Create volunteer application
     */
    public function create_application($data) {
        $data['application_date'] = date('Y-m-d H:i:s');
        $data['status'] = 'pending';
        return $this->db->insert($this->table, $data);
    }

    /**
     * Update volunteer
     */
    public function update_volunteer($id, $data) {
        $this->db->where('id', $id);
        return $this->db->update($this->table, $data);
    }

    /**
     * Approve volunteer
     */
    public function approve_volunteer($id, $admin_id) {
        $data = array(
            'status' => 'approved',
            'approved_by' => $admin_id,
            'approved_at' => date('Y-m-d H:i:s')
        );
        
        $this->db->where('id', $id);
        return $this->db->update($this->table, $data);
    }

    /**
     * Activate volunteer (make visible on website)
     */
    public function activate_volunteer($id) {
        $this->db->where('id', $id);
        return $this->db->update($this->table, array('status' => 'active'));
    }

    /**
     * Reject volunteer application
     */
    public function reject_volunteer($id, $admin_id) {
        $data = array(
            'status' => 'rejected',
            'approved_by' => $admin_id,
            'approved_at' => date('Y-m-d H:i:s')
        );
        
        $this->db->where('id', $id);
        return $this->db->update($this->table, $data);
    }

    /**
     * Delete volunteer
     */
    public function delete_volunteer($id) {
        $this->db->where('id', $id);
        return $this->db->delete($this->table);
    }

    /**
     * Check if email already exists
     */
    public function email_exists($email, $exclude_id = null) {
        $this->db->where('email', $email);
        
        if ($exclude_id) {
            $this->db->where('id !=', $exclude_id);
        }
        
        $query = $this->db->get($this->table);
        return $query->num_rows() > 0;
    }

    /**
     * Get volunteers statistics
     */
    public function get_statistics() {
        $stats = array();
        
        // Total volunteers
        $stats['total'] = $this->db->count_all($this->table);
        
        // By status
        $this->db->select('status, COUNT(*) as count');
        $this->db->from($this->table);
        $this->db->group_by('status');
        $query = $this->db->get();
        
        foreach ($query->result() as $row) {
            $stats[$row->status] = $row->count;
        }
        
        // Recent applications (last 30 days)
        $this->db->where('application_date >=', date('Y-m-d', strtotime('-30 days')));
        $stats['recent'] = $this->db->count_all_results($this->table);
        
        return $stats;
    }

    /**
     * Search volunteers
     */
    public function search_volunteers($keyword, $status = null, $limit = 10, $offset = 0) {
        $this->db->select('volunteers.*, admin_users.full_name as approved_by_name');
        $this->db->from($this->table);
        $this->db->join('admin_users', 'admin_users.id = volunteers.approved_by', 'left');
        
        // Search in multiple fields
        $this->db->group_start();
        $this->db->like('volunteers.full_name', $keyword);
        $this->db->or_like('volunteers.email', $keyword);
        $this->db->or_like('volunteers.phone', $keyword);
        $this->db->or_like('volunteers.skills', $keyword);
        $this->db->or_like('volunteers.occupation', $keyword);
        $this->db->group_end();
        
        if ($status) {
            $this->db->where('volunteers.status', $status);
        }
        
        $this->db->order_by('volunteers.application_date', 'DESC');
        $this->db->limit($limit, $offset);
        
        $query = $this->db->get();
        return $query->result();
    }

    /**
     * Get volunteers for export (CSV)
     */
    public function get_volunteers_for_export($status = null) {
        $this->db->select('volunteers.*, admin_users.full_name as approved_by_name');
        $this->db->from($this->table);
        $this->db->join('admin_users', 'admin_users.id = volunteers.approved_by', 'left');
        
        if ($status) {
            $this->db->where('volunteers.status', $status);
        }
        
        $this->db->order_by('volunteers.application_date', 'DESC');
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Add notes to volunteer
     */
    public function add_notes($id, $notes) {
        $this->db->where('id', $id);
        return $this->db->update($this->table, array('notes' => $notes));
    }
}