<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Registration Model
 * Handles French kid and adult learner registration data
 * Location: application/models/Registration_model.php
 * 
 * Database Tables:
 * - french_kid_learners (for kids registration)
 * - french_adult_learners (for adult registration)
 */
class Registration_model extends CI_Model {
    
    // Define table names as constants for easy maintenance
    private $kid_table = 'french_kid_learners';
    private $adult_table = 'french_adult_learners';
    
    public function __construct() {
        parent::__construct();
    }
    
    /**
     * Count registrations by type
     * @param string $type - 'kid', 'adult', or 'all'
     * @param string $status - 'Approved', 'Pending', 'Rejected', or 'all' (default: 'all')
     * @return int
     */
    public function count_registrations($type = 'all', $status = 'all') {
        if ($type === 'kid') {
            // Count from french_kid_learners table
            if ($status !== 'all') {
                $this->db->where('status', $status);
            }
            return $this->db->count_all_results($this->kid_table);
            
        } elseif ($type === 'adult') {
            // Count from french_adult_learners table
            if ($status !== 'all') {
                $this->db->where('status', $status);
            }
            return $this->db->count_all_results($this->adult_table);
            
        } else {
            // Count both
            $kids = $this->count_registrations('kid', $status);
            $adults = $this->count_registrations('adult', $status);
            return $kids + $adults;
        }
    }
    
    /**
     * Get all kid registrations
     * @param string $status - Filter by status (optional): 'Approved', 'Pending', 'Rejected'
     * @param int $limit - Limit results (optional)
     * @param int $offset - Offset for pagination (optional)
     * @return array
     */
    public function get_kid_registrations($status = null, $limit = null, $offset = 0) {
        if ($status) {
            $this->db->where('status', $status);
        }
        
        $this->db->order_by('created_at', 'DESC');
        
        if ($limit) {
            $this->db->limit($limit, $offset);
        }
        
        return $this->db->get($this->kid_table)->result();
    }
    
    /**
     * Get all adult registrations
     * @param string $status - Filter by status (optional): 'Approved', 'Pending', 'Rejected'
     * @param int $limit - Limit results (optional)
     * @param int $offset - Offset for pagination (optional)
     * @return array
     */
    public function get_adult_registrations($status = null, $limit = null, $offset = 0) {
        if ($status) {
            $this->db->where('status', $status);
        }
        
        $this->db->order_by('created_at', 'DESC');
        
        if ($limit) {
            $this->db->limit($limit, $offset);
        }
        
        return $this->db->get($this->adult_table)->result();
    }
    
    /**
     * Get single kid registration by ID
     * @param int $id
     * @return object|null
     */
    public function get_kid_registration($id) {
        return $this->db->get_where($this->kid_table, array('id' => $id))->row();
    }
    
    /**
     * Get single adult registration by ID
     * @param int $id
     * @return object|null
     */
    public function get_adult_registration($id) {
        return $this->db->get_where($this->adult_table, array('id' => $id))->row();
    }
    
    /**
     * Insert kid registration
     * @param array $data
     * @return int|bool - Insert ID or false
     */
    public function insert_kid_registration($data) {
        $data['created_at'] = date('Y-m-d H:i:s');
        $data['status'] = $data['status'] ?? 'Pending';
        
        if ($this->db->insert($this->kid_table, $data)) {
            return $this->db->insert_id();
        }
        return false;
    }
    
    /**
     * Insert adult registration
     * @param array $data
     * @return int|bool - Insert ID or false
     */
    public function insert_adult_registration($data) {
        $data['created_at'] = date('Y-m-d H:i:s');
        $data['status'] = $data['status'] ?? 'Pending';
        
        if ($this->db->insert($this->adult_table, $data)) {
            return $this->db->insert_id();
        }
        return false;
    }
    
    /**
     * Update kid registration
     * @param int $id
     * @param array $data
     * @return bool
     */
    public function update_kid_registration($id, $data) {
        $data['updated_at'] = date('Y-m-d H:i:s');
        $this->db->where('id', $id);
        return $this->db->update($this->kid_table, $data);
    }
    
    /**
     * Update adult registration
     * @param int $id
     * @param array $data
     * @return bool
     */
    public function update_adult_registration($id, $data) {
        $data['updated_at'] = date('Y-m-d H:i:s');
        $this->db->where('id', $id);
        return $this->db->update($this->adult_table, $data);
    }
    
    /**
     * Delete kid registration
     * @param int $id
     * @return bool
     */
    public function delete_kid_registration($id) {
        $this->db->where('id', $id);
        return $this->db->delete($this->kid_table);
    }
    
    /**
     * Delete adult registration
     * @param int $id
     * @return bool
     */
    public function delete_adult_registration($id) {
        $this->db->where('id', $id);
        return $this->db->delete($this->adult_table);
    }
    
    /**
     * Get registration statistics
     * @return array
     */
    public function get_statistics() {
        return array(
            'total_kids' => $this->count_registrations('kid'),
            'total_adults' => $this->count_registrations('adult'),
            'total_learners' => $this->count_registrations('all'),
            'pending_kids' => $this->count_registrations('kid', 'Pending'),
            'pending_adults' => $this->count_registrations('adult', 'Pending'),
            'approved_kids' => $this->count_registrations('kid', 'Approved'),
            'approved_adults' => $this->count_registrations('adult', 'Approved'),
            'rejected_kids' => $this->count_registrations('kid', 'Rejected'),
            'rejected_adults' => $this->count_registrations('adult', 'Rejected')
        );
    }
    
    /**
     * Check if email already registered (kid)
     * @param string $email
     * @return bool
     */
    public function is_kid_email_registered($email) {
        $this->db->where('email', $email);
        return $this->db->count_all_results($this->kid_table) > 0;
    }
    
    /**
     * Check if email already registered (adult)
     * @param string $email
     * @return bool
     */
    public function is_adult_email_registered($email) {
        $this->db->where('email', $email);
        return $this->db->count_all_results($this->adult_table) > 0;
    }
    
    /**
     * Get recent registrations (combined)
     * @param int $limit
     * @return array
     */
    public function get_recent_registrations($limit = 10) {
        // Get recent kids
        $kids = $this->db->select('id, first_name, surname, email, status, created_at, "kid" as type')
                         ->order_by('created_at', 'DESC')
                         ->limit($limit)
                         ->get($this->kid_table)
                         ->result_array();
        
        // Get recent adults
        $adults = $this->db->select('id, first_name, surname, email, status, created_at, "adult" as type')
                           ->order_by('created_at', 'DESC')
                           ->limit($limit)
                           ->get($this->adult_table)
                           ->result_array();
        
        // Merge and sort by created_at
        $combined = array_merge($kids, $adults);
        usort($combined, function($a, $b) {
            return strtotime($b['created_at']) - strtotime($a['created_at']);
        });
        
        // Return limited results
        return array_slice($combined, 0, $limit);
    }
    
    /**
     * Search registrations
     * @param string $keyword
     * @param string $type - 'kid', 'adult', or 'all'
     * @return array
     */
    public function search_registrations($keyword, $type = 'all') {
        $results = array();
        
        if ($type === 'kid' || $type === 'all') {
            $this->db->group_start();
            $this->db->like('first_name', $keyword);
            $this->db->or_like('surname', $keyword);
            $this->db->or_like('email', $keyword);
            $this->db->or_like('parent_name', $keyword);
            $this->db->group_end();
            
            $kids = $this->db->get($this->kid_table)->result();
            foreach ($kids as $kid) {
                $kid->type = 'kid';
                $results[] = $kid;
            }
        }
        
        if ($type === 'adult' || $type === 'all') {
            $this->db->group_start();
            $this->db->like('first_name', $keyword);
            $this->db->or_like('surname', $keyword);
            $this->db->or_like('email', $keyword);
            $this->db->group_end();
            
            $adults = $this->db->get($this->adult_table)->result();
            foreach ($adults as $adult) {
                $adult->type = 'adult';
                $results[] = $adult;
            }
        }
        
        return $results;
    }
    
    /**
     * Get learners by French level
     * @param string $type - 'kid', 'adult', or 'all'
     * @param string $level - 'Beginner', 'BeginnerPlus', 'Intermediate', 'Almost advanced'
     * @return array
     */
    public function get_by_french_level($type = 'all', $level = null) {
        $results = array();
        
        if ($type === 'kid' || $type === 'all') {
            if ($level) {
                $this->db->where('french_level', $level);
            }
            $kids = $this->db->get($this->kid_table)->result();
            foreach ($kids as $kid) {
                $kid->type = 'kid';
                $results[] = $kid;
            }
        }
        
        if ($type === 'adult' || $type === 'all') {
            if ($level) {
                $this->db->where('french_level', $level);
            }
            $adults = $this->db->get($this->adult_table)->result();
            foreach ($adults as $adult) {
                $adult->type = 'adult';
                $results[] = $adult;
            }
        }
        
        return $results;
    }
    
    /**
     * Update registration status
     * @param string $type - 'kid' or 'adult'
     * @param int $id
     * @param string $status - 'Pending', 'Approved', 'Rejected'
     * @return bool
     */
    public function update_status($type, $id, $status) {
        $data = array(
            'status' => $status,
            'updated_at' => date('Y-m-d H:i:s')
        );
        
        $table = ($type === 'kid') ? $this->kid_table : $this->adult_table;
        
        $this->db->where('id', $id);
        return $this->db->update($table, $data);
    }
}