<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Media_model extends CI_Model {
    private $table = 'media';

    /**
     * Get media items with optional filtering
     * @param int|null $category_id Category filter
     * @param string|null $media_type Media type filter
     * @param string|null $status Status filter
     * @return array Media items
     */
  public function get_media($category_id = NULL, $media_type = NULL, $status = 'published') {
    // Start query
    $this->db->select('m.*, mc.name as category_name')
             ->from($this->table . ' m')
             ->join('media_categories mc', 'm.category_id = mc.id', 'left');

    // Apply filters only if they are not NULL
    if ($category_id !== NULL && $category_id !== '') {
        $this->db->where('m.category_id', $category_id);
    }

    if ($media_type !== NULL && $media_type !== '') {
        $this->db->where('m.file_type', $media_type);
    }

    if ($status !== NULL && $status !== '') {
        $this->db->where('m.status', $status);
    }

    // Order by most recent first
    $this->db->order_by('m.created_at', 'DESC');

    // Execute and return
    $query = $this->db->get();
    
    // Log the query for debugging
    log_message('debug', 'Media Query: ' . $this->db->last_query());
    
    return $query->result();
}

  /**
     * Get a specific media item by ID
     * @param int $id Media item ID
     * @return object|null Media item
     */
    public function get_media_by_id($id) {
        return $this->db->select('m.*, mc.name as category_name')
                        ->from($this->table . ' m')
                        ->join('media_categories mc', 'm.category_id = mc.id', 'left')
                        ->where('m.id', $id)
                        ->get()
                        ->row();
    }

    /**
     * Get related media items
     * @param int $category_id Category ID
     * @param int $exclude_id ID of current media item to exclude
     * @param int $limit Number of related items to fetch
     * @return array Related media items
     */
    public function get_related_media($category_id, $exclude_id, $limit = 4) {
        return $this->db->where('category_id', $category_id)
                        ->where('id !=', $exclude_id)
                        ->where('status', 'published')
                        ->limit($limit)
                        ->order_by('RAND()')
                        ->get($this->table)
                        ->result();
    }

    /**
     * Insert new media item
     * @param array $data Media item data
     * @return int|bool Inserted ID or FALSE
     */
    public function insert_media($data) {
        $this->db->trans_start();
        
        $this->db->insert($this->table, $data);
        $insert_id = $this->db->insert_id();
        
        $this->db->trans_complete();
        
        return $this->db->trans_status() ? $insert_id : FALSE;
    }

    /**
     * Update media item
     * @param int $id Media item ID
     * @param array $data Updated media data
     * @return bool Success status
     */
    public function update_media($id, $data) {
        return $this->db->where('id', $id)
                        ->update($this->table, $data);
    }

    /**
     * Delete media item
     * @param int $id Media item ID
     * @return bool Success status
     */
    public function delete_media($id) {
        // First, get the file path to potentially delete the physical file
        $media_item = $this->get_media_by_id($id);
        
        $this->db->trans_start();
        
        // Delete from database
        $this->db->where('id', $id)
                 ->delete($this->table);
        
        $this->db->trans_complete();
        
        // If transaction successful, attempt to delete physical file
        if ($this->db->trans_status()) {
            $this->_delete_file($media_item->file_path);
            return TRUE;
        }
        
        return FALSE;
    }

    /**
     * Delete physical media file
     * @param string $file_path Path to file
     */
    private function _delete_file($file_path) {
        $full_path = FCPATH . 'assets/uploads/media/' . $file_path;
        
        if (file_exists($full_path)) {
            @unlink($full_path);
        }
    }

    /**
     * Get media statistics
     * @return array Media statistics
     */
    public function get_media_statistics() {
        $stats = [];
        
        // Total media count
        $stats['total_media'] = $this->db->count_all($this->table);
        
        // Media by type
        $stats['media_by_type'] = $this->db->select('file_type, COUNT(*) as count')
                                           ->group_by('file_type')
                                           ->get($this->table)
                                           ->result();
        
        // Media by category
        $stats['media_by_category'] = $this->db->select('c.name as category_name, COUNT(m.id) as count')
                                               ->from($this->table . ' m')
                                               ->join('media_categories c', 'm.category_id = c.id', 'left')
                                               ->group_by('c.name')
                                               ->get()
                                               ->result();
        
        return $stats;
    }
}