<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Contact Model
 * Handles all contact message database operations
 * Location: application/models/Contact_model.php
 */
class Contact_model extends CI_Model {

    private $table = 'contact_messages';

    public function __construct() {
        parent::__construct();
    }

    /**
     * Get all messages with pagination
     */
    public function get_messages($limit = 10, $offset = 0, $status = null) {
        $this->db->select('contact_messages.*, admin_users.full_name as replied_by_name');
        $this->db->from($this->table);
        $this->db->join('admin_users', 'admin_users.id = contact_messages.replied_by', 'left');
        
        if ($status) {
            $this->db->where('contact_messages.status', $status);
        }
        
        $this->db->order_by('contact_messages.created_at', 'DESC');
        $this->db->limit($limit, $offset);
        
        $query = $this->db->get();
        return $query->result();
    }

    /**
     * Count messages
     */
    public function count_messages($status = null) {
        if ($status) {
            $this->db->where('status', $status);
        }
        return $this->db->count_all_results($this->table);
    }

    /**
     * Get message by ID
     */
    public function get_message($id) {
        $this->db->select('contact_messages.*, admin_users.full_name as replied_by_name, admin_users.email as replied_by_email');
        $this->db->from($this->table);
        $this->db->join('admin_users', 'admin_users.id = contact_messages.replied_by', 'left');
        $this->db->where('contact_messages.id', $id);
        
        $query = $this->db->get();
        return $query->row();
    }

    /**
     * Create contact message
     */
    public function create_message($data) {
        $data['status'] = 'new';
        $data['ip_address'] = $this->input->ip_address();
        return $this->db->insert($this->table, $data);
    }

    /**
     * Get last insert ID
     */
    public function get_last_insert_id() {
        return $this->db->insert_id();
    }

    /**
     * Mark message as read
     */
    public function mark_as_read($id) {
        $this->db->where('id', $id);
        return $this->db->update($this->table, array('status' => 'read'));
    }

    /**
     * Mark message as replied
     */
    public function mark_as_replied($id, $admin_id, $reply_message) {
        $data = array(
            'status' => 'replied',
            'replied_by' => $admin_id,
            'replied_at' => date('Y-m-d H:i:s'),
            'reply_message' => $reply_message
        );
        
        $this->db->where('id', $id);
        return $this->db->update($this->table, $data);
    }

    /**
     * Archive message
     */
    public function archive_message($id) {
        $this->db->where('id', $id);
        return $this->db->update($this->table, array('status' => 'archived'));
    }

    /**
     * Delete message
     */
    public function delete_message($id) {
        $this->db->where('id', $id);
        return $this->db->delete($this->table);
    }

    /**
     * Get message statistics
     */
    public function get_statistics() {
        $stats = array();
        
        // Total messages
        $stats['total'] = $this->db->count_all($this->table);
        
        // By status
        $this->db->select('status, COUNT(*) as count');
        $this->db->from($this->table);
        $this->db->group_by('status');
        $query = $this->db->get();
        
        foreach ($query->result() as $row) {
            $stats[$row->status] = $row->count;
        }
        
        // Recent messages (last 30 days)
        $this->db->where('created_at >=', date('Y-m-d', strtotime('-30 days')));
        $stats['recent'] = $this->db->count_all_results($this->table);
        
        return $stats;
    }

    /**
     * Get unread messages count
     */
    public function get_unread_count() {
        $this->db->where('status', 'new');
        return $this->db->count_all_results($this->table);
    }

    /**
     * Search messages
     */
    public function search_messages($keyword, $status = null, $limit = 10, $offset = 0) {
        $this->db->select('contact_messages.*, admin_users.full_name as replied_by_name');
        $this->db->from($this->table);
        $this->db->join('admin_users', 'admin_users.id = contact_messages.replied_by', 'left');
        
        // Search in multiple fields
        $this->db->group_start();
        $this->db->like('contact_messages.name', $keyword);
        $this->db->or_like('contact_messages.email', $keyword);
        $this->db->or_like('contact_messages.subject', $keyword);
        $this->db->or_like('contact_messages.message', $keyword);
        $this->db->group_end();
        
        if ($status) {
            $this->db->where('contact_messages.status', $status);
        }
        
        $this->db->order_by('contact_messages.created_at', 'DESC');
        $this->db->limit($limit, $offset);
        
        $query = $this->db->get();
        return $query->result();
    }

    /**
     * Get messages for export (CSV)
     */
    public function get_messages_for_export($status = null) {
        $this->db->select('contact_messages.*, admin_users.full_name as replied_by_name');
        $this->db->from($this->table);
        $this->db->join('admin_users', 'admin_users.id = contact_messages.replied_by', 'left');
        
        if ($status) {
            $this->db->where('contact_messages.status', $status);
        }
        
        $this->db->order_by('contact_messages.created_at', 'DESC');
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Bulk update status
     */
    public function bulk_update_status($ids, $status) {
        $this->db->where_in('id', $ids);
        return $this->db->update($this->table, array('status' => $status));
    }

    /**
     * Bulk delete
     */
    public function bulk_delete($ids) {
        $this->db->where_in('id', $ids);
        return $this->db->delete($this->table);
    }
}