<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Consultation Model
 * Handles all consultation request database operations
 * Location: application/models/Consultation_model.php
 */
class Consultation_model extends CI_Model {

    private $table = 'consultation_requests';

    public function __construct() {
        parent::__construct();
    }

    /**
     * Get all consultation requests with pagination
     */
    public function get_consultations($limit = 10, $offset = 0, $status = null) {
        $this->db->select('consultation_requests.*, admin_users.full_name as assigned_to_name');
        $this->db->from($this->table);
        $this->db->join('admin_users', 'admin_users.id = consultation_requests.assigned_to', 'left');
        
        if ($status) {
            $this->db->where('consultation_requests.status', $status);
        }
        
        $this->db->order_by('consultation_requests.created_at', 'DESC');
        $this->db->limit($limit, $offset);
        
        $query = $this->db->get();
        return $query->result();
    }

    /**
     * Count consultation requests
     */
    public function count_consultations($status = null) {
        if ($status) {
            $this->db->where('status', $status);
        }
        return $this->db->count_all_results($this->table);
    }

    /**
     * Get consultation by ID
     */
    public function get_consultation($id) {
        $this->db->select('consultation_requests.*, admin_users.full_name as assigned_to_name, admin_users.email as assigned_to_email');
        $this->db->from($this->table);
        $this->db->join('admin_users', 'admin_users.id = consultation_requests.assigned_to', 'left');
        $this->db->where('consultation_requests.id', $id);
        
        $query = $this->db->get();
        return $query->row();
    }

    /**
     * Create consultation request
     */
    public function create_consultation($data) {
        $data['status'] = 'new';
        return $this->db->insert($this->table, $data);
    }

    /**
     * Get last insert ID
     */
    public function get_last_insert_id() {
        return $this->db->insert_id();
    }

    /**
     * Update consultation status
     */
    public function update_status($id, $status, $data = array()) {
        $data['status'] = $status;
        
        $this->db->where('id', $id);
        return $this->db->update($this->table, $data);
    }

    /**
     * Assign consultation to admin
     */
    public function assign_consultation($id, $admin_id) {
        $data = array(
            'assigned_to' => $admin_id,
            'status' => 'contacted'
        );
        
        $this->db->where('id', $id);
        return $this->db->update($this->table, $data);
    }

    /**
     * Schedule consultation
     */
    public function schedule_consultation($id, $scheduled_date, $admin_id = null) {
        $data = array(
            'scheduled_date' => $scheduled_date,
            'status' => 'scheduled'
        );
        
        if ($admin_id) {
            $data['assigned_to'] = $admin_id;
        }
        
        $this->db->where('id', $id);
        return $this->db->update($this->table, $data);
    }

    /**
     * Update consultation
     */
    public function update_consultation($id, $data) {
        $this->db->where('id', $id);
        return $this->db->update($this->table, $data);
    }

    /**
     * Delete consultation
     */
    public function delete_consultation($id) {
        $this->db->where('id', $id);
        return $this->db->delete($this->table);
    }

    /**
     * Add notes to consultation
     */
    public function add_notes($id, $notes) {
        $this->db->where('id', $id);
        return $this->db->update($this->table, array('notes' => $notes));
    }

    /**
     * Get consultation statistics
     */
    public function get_statistics() {
        $stats = array();
        
        // Total consultations
        $stats['total'] = $this->db->count_all($this->table);
        
        // By status
        $this->db->select('status, COUNT(*) as count');
        $this->db->from($this->table);
        $this->db->group_by('status');
        $query = $this->db->get();
        
        foreach ($query->result() as $row) {
            $stats[$row->status] = $row->count;
        }
        
        // Recent requests (last 30 days)
        $this->db->where('created_at >=', date('Y-m-d', strtotime('-30 days')));
        $stats['recent'] = $this->db->count_all_results($this->table);
        
        // Upcoming scheduled
        $this->db->where('status', 'scheduled');
        $this->db->where('scheduled_date >=', date('Y-m-d'));
        $stats['upcoming'] = $this->db->count_all_results($this->table);
        
        return $stats;
    }

    /**
     * Get upcoming consultations
     */
    public function get_upcoming_consultations($limit = 10) {
        $this->db->select('consultation_requests.*, admin_users.full_name as assigned_to_name');
        $this->db->from($this->table);
        $this->db->join('admin_users', 'admin_users.id = consultation_requests.assigned_to', 'left');
        $this->db->where('consultation_requests.status', 'scheduled');
        $this->db->where('consultation_requests.scheduled_date >=', date('Y-m-d'));
        $this->db->order_by('consultation_requests.scheduled_date', 'ASC');
        $this->db->limit($limit);
        
        $query = $this->db->get();
        return $query->result();
    }

    /**
     * Search consultations
     */
    public function search_consultations($keyword, $status = null, $limit = 10, $offset = 0) {
        $this->db->select('consultation_requests.*, admin_users.full_name as assigned_to_name');
        $this->db->from($this->table);
        $this->db->join('admin_users', 'admin_users.id = consultation_requests.assigned_to', 'left');
        
        // Search in multiple fields
        $this->db->group_start();
        $this->db->like('consultation_requests.full_name', $keyword);
        $this->db->or_like('consultation_requests.email', $keyword);
        $this->db->or_like('consultation_requests.phone', $keyword);
        $this->db->or_like('consultation_requests.subject', $keyword);
        $this->db->or_like('consultation_requests.organization', $keyword);
        $this->db->group_end();
        
        if ($status) {
            $this->db->where('consultation_requests.status', $status);
        }
        
        $this->db->order_by('consultation_requests.created_at', 'DESC');
        $this->db->limit($limit, $offset);
        
        $query = $this->db->get();
        return $query->result();
    }

    /**
     * Get consultations for export (CSV)
     */
    public function get_consultations_for_export($status = null) {
        $this->db->select('consultation_requests.*, admin_users.full_name as assigned_to_name');
        $this->db->from($this->table);
        $this->db->join('admin_users', 'admin_users.id = consultation_requests.assigned_to', 'left');
        
        if ($status) {
            $this->db->where('consultation_requests.status', $status);
        }
        
        $this->db->order_by('consultation_requests.created_at', 'DESC');
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get consultations by type
     */
    public function get_by_type($type, $limit = 10, $offset = 0) {
        $this->db->select('consultation_requests.*, admin_users.full_name as assigned_to_name');
        $this->db->from($this->table);
        $this->db->join('admin_users', 'admin_users.id = consultation_requests.assigned_to', 'left');
        $this->db->where('consultation_requests.consultation_type', $type);
        $this->db->order_by('consultation_requests.created_at', 'DESC');
        $this->db->limit($limit, $offset);
        
        $query = $this->db->get();
        return $query->result();
    }

    /**
     * Check if email has pending consultation
     */
    public function has_pending_consultation($email) {
        $this->db->where('email', $email);
        $this->db->where_in('status', array('new', 'contacted', 'scheduled'));
        $query = $this->db->get($this->table);
        return $query->num_rows() > 0;
    }
}