<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Custom Upload Library
 * Extends CI_Upload with additional functionality
 * Location: application/libraries/MY_Upload.php
 * 
 * Usage in controller:
 * $this->load->library('upload');
 * $result = $this->upload->upload_image($field_name, $upload_path);
 */

class MY_Upload extends CI_Upload {

    public function __construct($config = array()) {
        parent::__construct($config);
    }

    /**
     * Upload image file
     * 
     * @param string $field_name Form field name
     * @param string $upload_path Upload directory path
     * @param array $custom_config Custom configuration
     * @return array Result with status and data
     */
    public function upload_image($field_name = 'file', $upload_path = './uploads/images/', $custom_config = array()) {
        // Create directory if not exists
        if (!is_dir($upload_path)) {
            mkdir($upload_path, 0777, true);
        }

        // Default configuration
        $config = array(
            'upload_path' => $upload_path,
            'allowed_types' => 'jpg|jpeg|png|gif|webp',
            'max_size' => 5120, // 5MB
            'max_width' => 3000,
            'max_height' => 3000,
            'encrypt_name' => FALSE,
            'remove_spaces' => TRUE,
            'file_ext_tolower' => TRUE
        );

        // Merge custom config
        $config = array_merge($config, $custom_config);

        // Generate unique filename
        if (!empty($_FILES[$field_name]['name'])) {
            $CI =& get_instance();
            $CI->load->helper('MY_utility');
            $config['file_name'] = generate_unique_filename($_FILES[$field_name]['name']);
        }

        $this->initialize($config);

        if ($this->do_upload($field_name)) {
            $upload_data = $this->data();
            
            return array(
                'status' => true,
                'message' => 'File uploaded successfully',
                'data' => $upload_data,
                'file_path' => $upload_data['file_name']
            );
        } else {
            return array(
                'status' => false,
                'message' => $this->display_errors('', ''),
                'data' => null,
                'file_path' => null
            );
        }
    }

    /**
     * Upload document file
     * 
     * @param string $field_name Form field name
     * @param string $upload_path Upload directory path
     * @param array $custom_config Custom configuration
     * @return array Result with status and data
     */
    public function upload_document($field_name = 'file', $upload_path = './uploads/documents/', $custom_config = array()) {
        // Create directory if not exists
        if (!is_dir($upload_path)) {
            mkdir($upload_path, 0777, true);
        }

        // Default configuration
        $config = array(
            'upload_path' => $upload_path,
            'allowed_types' => 'pdf|doc|docx|xls|xlsx|ppt|pptx|txt',
            'max_size' => 10240, // 10MB
            'encrypt_name' => FALSE,
            'remove_spaces' => TRUE,
            'file_ext_tolower' => TRUE
        );

        // Merge custom config
        $config = array_merge($config, $custom_config);

        // Generate unique filename
        if (!empty($_FILES[$field_name]['name'])) {
            $CI =& get_instance();
            $CI->load->helper('MY_utility');
            $config['file_name'] = generate_unique_filename($_FILES[$field_name]['name']);
        }

        $this->initialize($config);

        if ($this->do_upload($field_name)) {
            $upload_data = $this->data();
            
            return array(
                'status' => true,
                'message' => 'File uploaded successfully',
                'data' => $upload_data,
                'file_path' => $upload_data['file_name']
            );
        } else {
            return array(
                'status' => false,
                'message' => $this->display_errors('', ''),
                'data' => null,
                'file_path' => null
            );
        }
    }

    /**
     * Upload profile photo
     * 
     * @param string $field_name Form field name
     * @param string $upload_path Upload directory path
     * @return array Result with status and data
     */
    public function upload_profile_photo($field_name = 'profile_photo', $upload_path = './uploads/profiles/') {
        $custom_config = array(
            'max_size' => 2048, // 2MB
            'max_width' => 1000,
            'max_height' => 1000,
            'allowed_types' => 'jpg|jpeg|png'
        );

        return $this->upload_image($field_name, $upload_path, $custom_config);
    }

    /**
     * Upload blog featured image
     * 
     * @param string $field_name Form field name
     * @param string $upload_path Upload directory path
     * @return array Result with status and data
     */
    public function upload_blog_image($field_name = 'featured_image', $upload_path = './uploads/blog/') {
        $custom_config = array(
            'max_size' => 3072, // 3MB
            'max_width' => 1920,
            'max_height' => 1080
        );

        return $this->upload_image($field_name, $upload_path, $custom_config);
    }

    /**
     * Upload carousel media
     * 
     * @param string $field_name Form field name
     * @param string $upload_path Upload directory path
     * @return array Result with status and data
     */
    public function upload_carousel_media($field_name = 'media', $upload_path = './uploads/carousel/') {
        // Create directory if not exists
        if (!is_dir($upload_path)) {
            mkdir($upload_path, 0777, true);
        }

        $config = array(
            'upload_path' => $upload_path,
            'allowed_types' => 'jpg|jpeg|png|gif|webp|mp4|mov|avi',
            'max_size' => 20480, // 20MB for video
            'encrypt_name' => FALSE,
            'remove_spaces' => TRUE,
            'file_ext_tolower' => TRUE
        );

        // Generate unique filename
        if (!empty($_FILES[$field_name]['name'])) {
            $CI =& get_instance();
            $CI->load->helper('MY_utility');
            $config['file_name'] = generate_unique_filename($_FILES[$field_name]['name']);
        }

        $this->initialize($config);

        if ($this->do_upload($field_name)) {
            $upload_data = $this->data();
            
            return array(
                'status' => true,
                'message' => 'File uploaded successfully',
                'data' => $upload_data,
                'file_path' => $upload_data['file_name'],
                'media_type' => $this->is_video($upload_data['file_ext']) ? 'video' : 'image'
            );
        } else {
            return array(
                'status' => false,
                'message' => $this->display_errors('', ''),
                'data' => null,
                'file_path' => null,
                'media_type' => null
            );
        }
    }

    /**
     * Check if file extension is video
     * 
     * @param string $ext File extension
     * @return bool Is video
     */
    private function is_video($ext) {
        $video_extensions = array('.mp4', '.mov', '.avi', '.wmv', '.flv', '.webm');
        return in_array(strtolower($ext), $video_extensions);
    }

    /**
     * Delete uploaded file
     * 
     * @param string $file_path File path to delete
     * @return bool Success status
     */
    public function delete_file($file_path) {
        if (file_exists($file_path)) {
            return unlink($file_path);
        }
        return false;
    }

    /**
     * Resize image
     * 
     * @param string $source_path Source image path
     * @param int $width New width
     * @param int $height New height
     * @param bool $maintain_ratio Maintain aspect ratio
     * @return bool Success status
     */
    public function resize_image($source_path, $width, $height, $maintain_ratio = TRUE) {
        $CI =& get_instance();
        $CI->load->library('image_lib');

        $config = array(
            'source_image' => $source_path,
            'maintain_ratio' => $maintain_ratio,
            'width' => $width,
            'height' => $height
        );

        $CI->image_lib->initialize($config);

        if ($CI->image_lib->resize()) {
            $CI->image_lib->clear();
            return true;
        } else {
            return false;
        }
    }

    /**
     * Create thumbnail
     * 
     * @param string $source_path Source image path
     * @param int $width Thumbnail width
     * @param int $height Thumbnail height
     * @param string $thumb_path Thumbnail save path
     * @return bool Success status
     */
    public function create_thumbnail($source_path, $width = 150, $height = 150, $thumb_path = null) {
        $CI =& get_instance();
        $CI->load->library('image_lib');

        if (!$thumb_path) {
            $path_info = pathinfo($source_path);
            $thumb_path = $path_info['dirname'] . '/thumb_' . $path_info['basename'];
        }

        $config = array(
            'source_image' => $source_path,
            'new_image' => $thumb_path,
            'maintain_ratio' => TRUE,
            'width' => $width,
            'height' => $height
        );

        $CI->image_lib->initialize($config);

        if ($CI->image_lib->resize()) {
            $CI->image_lib->clear();
            return $thumb_path;
        } else {
            return false;
        }
    }
}