<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Authentication Library
 * Handles admin login, logout, and session management
 * Location: application/libraries/Auth_library.php
 * 
 * Usage in controller:
 * $this->load->library('auth_library');
 * $this->auth_library->require_login();
 */

class Auth_library {

    protected $CI;

    public function __construct() {
        $this->CI =& get_instance();
        $this->CI->load->model('Common_model');
    }

    /**
     * Login admin user
     * 
     * @param string $username Username or email
     * @param string $password Password
     * @return array Result with status and message
     */
    public function login($username, $password) {
        // Try to get user by username first
        $user = $this->CI->Common_model->get_admin_by_username($username);
        
        // If not found, try by email
        if (!$user) {
            $user = $this->CI->Common_model->get_admin_by_email($username);
        }

        // Check if user exists
        if (!$user) {
            return array(
                'status' => false,
                'message' => 'Invalid username/email or password'
            );
        }

        // Check if account is active
        if ($user->status != 'active') {
            return array(
                'status' => false,
                'message' => 'Your account has been deactivated. Please contact administrator.'
            );
        }

        // Verify password
        if (!password_verify($password, $user->password)) {
            return array(
                'status' => false,
                'message' => 'Invalid username/email or password'
            );
        }

        // Set session data
        $session_data = array(
            'admin_logged_in' => TRUE,
            'admin_data' => array(
                'id' => $user->id,
                'username' => $user->username,
                'email' => $user->email,
                'full_name' => $user->full_name,
                'role' => $user->role
            )
        );

        $this->CI->session->set_userdata($session_data);

        // Update last login
        $this->CI->Common_model->update_last_login($user->id);

        return array(
            'status' => true,
            'message' => 'Login successful',
            'redirect' => base_url('admin/dashboard')
        );
    }

    /**
     * Logout admin user
     * 
     * @return bool Success status
     */
    public function logout() {
        $this->CI->session->unset_userdata('admin_logged_in');
        $this->CI->session->unset_userdata('admin_data');
        $this->CI->session->sess_destroy();
        return true;
    }

    /**
     * Check if admin is logged in
     * 
     * @return bool Login status
     */
    public function is_logged_in() {
        return $this->CI->session->userdata('admin_logged_in') === TRUE;
    }

    /**
     * Require login - redirect to login if not logged in
     * 
     * @param string $redirect_url URL to redirect after login
     */
    public function require_login($redirect_url = null) {
        if (!$this->is_logged_in()) {
            if ($redirect_url) {
                $this->CI->session->set_userdata('login_redirect', $redirect_url);
            }
            redirect('admin/login');
        }
    }

    /**
     * Get logged in admin data
     * 
     * @param string $key Specific key to retrieve
     * @return mixed Admin data
     */
    public function get_admin_data($key = null) {
        $admin_data = $this->CI->session->userdata('admin_data');
        
        if ($key) {
            return isset($admin_data[$key]) ? $admin_data[$key] : null;
        }
        
        return $admin_data;
    }

    /**
     * Get admin ID
     * 
     * @return int Admin ID
     */
    public function get_admin_id() {
        return $this->get_admin_data('id');
    }

    /**
     * Get admin role
     * 
     * @return string Admin role
     */
    public function get_admin_role() {
        return $this->get_admin_data('role');
    }

    /**
     * Check if admin has specific role
     * 
     * @param string|array $roles Role(s) to check
     * @return bool Has role
     */
    public function has_role($roles) {
        $admin_role = $this->get_admin_role();
        
        if (is_array($roles)) {
            return in_array($admin_role, $roles);
        }
        
        return $admin_role == $roles;
    }

    /**
     * Require specific role
     * 
     * @param string|array $roles Required role(s)
     * @param string $redirect_url URL to redirect if unauthorized
     */
    public function require_role($roles, $redirect_url = 'admin/dashboard') {
        $this->require_login();
        
        if (!$this->has_role($roles)) {
            $this->CI->load->helper('MY_utility');
            set_flash_message('error', 'You do not have permission to access this page.');
            redirect($redirect_url);
        }
    }

    /**
     * Check permission for action
     * 
     * @param string $permission Permission to check (view, create, edit, delete)
     * @return bool Has permission
     */
    public function has_permission($permission) {
        $role = $this->get_admin_role();
        
        // Super admin has all permissions
        if ($role == 'super_admin') {
            return true;
        }
        
        // Define role permissions
        $permissions = array(
            'admin' => array('view', 'create', 'edit', 'delete'),
            'editor' => array('view', 'create', 'edit')
        );
        
        if (isset($permissions[$role]) && in_array($permission, $permissions[$role])) {
            return true;
        }
        
        return false;
    }

    /**
     * Require permission for action
     * 
     * @param string $permission Required permission
     * @param string $redirect_url URL to redirect if unauthorized
     */
    public function require_permission($permission, $redirect_url = null) {
        $this->require_login();
        
        if (!$this->has_permission($permission)) {
            $this->CI->load->helper('MY_utility');
            set_flash_message('error', 'You do not have permission to perform this action.');
            
            if ($redirect_url) {
                redirect($redirect_url);
            } else {
                redirect($_SERVER['HTTP_REFERER']);
            }
        }
    }

    /**
     * Change password
     * 
     * @param int $admin_id Admin ID
     * @param string $old_password Old password
     * @param string $new_password New password
     * @return array Result with status and message
     */
    public function change_password($admin_id, $old_password, $new_password) {
        // Get admin data
        $admin = $this->CI->Common_model->get_admin($admin_id);
        
        if (!$admin) {
            return array(
                'status' => false,
                'message' => 'Admin not found'
            );
        }

        // Verify old password
        if (!password_verify($old_password, $admin->password)) {
            return array(
                'status' => false,
                'message' => 'Current password is incorrect'
            );
        }

        // Update password
        if ($this->CI->Common_model->update_admin_password($admin_id, $new_password)) {
            return array(
                'status' => true,
                'message' => 'Password changed successfully'
            );
        } else {
            return array(
                'status' => false,
                'message' => 'Failed to update password. Please try again.'
            );
        }
    }

    /**
     * Generate password reset token
     * 
     * @param string $email Admin email
     * @return array Result with status and token
     */
    public function generate_reset_token($email) {
        $admin = $this->CI->Common_model->get_admin_by_email($email);
        
        if (!$admin) {
            return array(
                'status' => false,
                'message' => 'Email address not found'
            );
        }

        // Generate token
        $this->CI->load->helper('MY_utility');
        $token = generate_random_string(32);
        $expiry = date('Y-m-d H:i:s', strtotime('+1 hour'));

        // Store token in session (in production, store in database)
        $this->CI->session->set_userdata('reset_token_' . $admin->id, array(
            'token' => $token,
            'expiry' => $expiry
        ));

        return array(
            'status' => true,
            'token' => $token,
            'admin_id' => $admin->id,
            'message' => 'Password reset token generated'
        );
    }

    /**
     * Verify reset token
     * 
     * @param int $admin_id Admin ID
     * @param string $token Reset token
     * @return bool Valid token
     */
    public function verify_reset_token($admin_id, $token) {
        $reset_data = $this->CI->session->userdata('reset_token_' . $admin_id);
        
        if (!$reset_data) {
            return false;
        }

        // Check if token matches and not expired
        if ($reset_data['token'] == $token && strtotime($reset_data['expiry']) > time()) {
            return true;
        }

        return false;
    }

    /**
     * Reset password with token
     * 
     * @param int $admin_id Admin ID
     * @param string $token Reset token
     * @param string $new_password New password
     * @return array Result with status and message
     */
    public function reset_password($admin_id, $token, $new_password) {
        if (!$this->verify_reset_token($admin_id, $token)) {
            return array(
                'status' => false,
                'message' => 'Invalid or expired reset token'
            );
        }

        // Update password
        if ($this->CI->Common_model->update_admin_password($admin_id, $new_password)) {
            // Remove token from session
            $this->CI->session->unset_userdata('reset_token_' . $admin_id);
            
            return array(
                'status' => true,
                'message' => 'Password reset successfully'
            );
        } else {
            return array(
                'status' => false,
                'message' => 'Failed to reset password. Please try again.'
            );
        }
    }
}