<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Custom Utility Helper Functions
 * Location: application/helpers/MY_utility_helper.php
 * 
 * Load in controller: $this->load->helper('MY_utility');
 */

if (!function_exists('format_date')) {
    /**
     * Format date for display
     * 
     * @param string $date Date string
     * @param string $format Date format
     * @return string Formatted date
     */
    function format_date($date, $format = 'F j, Y') {
        if (empty($date) || $date == '0000-00-00' || $date == '0000-00-00 00:00:00') {
            return 'N/A';
        }
        return date($format, strtotime($date));
    }
}

if (!function_exists('format_datetime')) {
    /**
     * Format datetime for display
     * 
     * @param string $datetime Datetime string
     * @return string Formatted datetime
     */
    function format_datetime($datetime) {
        return format_date($datetime, 'F j, Y g:i A');
    }
}

if (!function_exists('time_ago')) {
    /**
     * Convert datetime to "time ago" format
     * 
     * @param string $datetime Datetime string
     * @return string Time ago string
     */
    function time_ago($datetime) {
        $timestamp = strtotime($datetime);
        $difference = time() - $timestamp;
        
        $periods = array(
            'year' => 31536000,
            'month' => 2592000,
            'week' => 604800,
            'day' => 86400,
            'hour' => 3600,
            'minute' => 60,
            'second' => 1
        );
        
        foreach ($periods as $key => $value) {
            if ($difference >= $value) {
                $time = floor($difference / $value);
                $difference %= $value;
                return $time . ' ' . $key . ($time > 1 ? 's' : '') . ' ago';
            }
        }
        
        return 'Just now';
    }
}

if (!function_exists('truncate_text')) {
    /**
     * Truncate text to specific length
     * 
     * @param string $text Text to truncate
     * @param int $length Maximum length
     * @param string $suffix Suffix to add
     * @return string Truncated text
     */
    function truncate_text($text, $length = 100, $suffix = '...') {
        if (strlen($text) > $length) {
            return substr($text, 0, $length) . $suffix;
        }
        return $text;
    }
}

if (!function_exists('generate_random_string')) {
    /**
     * Generate random string
     * 
     * @param int $length String length
     * @return string Random string
     */
    function generate_random_string($length = 10) {
        $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $random_string = '';
        
        for ($i = 0; $i < $length; $i++) {
            $random_string .= $characters[rand(0, strlen($characters) - 1)];
        }
        
        return $random_string;
    }
}

if (!function_exists('sanitize_filename')) {
    /**
     * Sanitize filename for upload
     * 
     * @param string $filename Original filename
     * @return string Sanitized filename
     */
    function sanitize_filename($filename) {
        $filename = preg_replace('/[^a-zA-Z0-9._-]/', '_', $filename);
        $filename = preg_replace('/_+/', '_', $filename);
        return strtolower($filename);
    }
}

if (!function_exists('generate_unique_filename')) {
    /**
     * Generate unique filename with timestamp
     * 
     * @param string $filename Original filename
     * @return string Unique filename
     */
    function generate_unique_filename($filename) {
        $ext = pathinfo($filename, PATHINFO_EXTENSION);
        $name = pathinfo($filename, PATHINFO_FILENAME);
        $name = sanitize_filename($name);
        return $name . '_' . time() . '.' . $ext;
    }
}

if (!function_exists('format_file_size')) {
    /**
     * Format file size in human readable format
     * 
     * @param int $bytes File size in bytes
     * @return string Formatted file size
     */
    function format_file_size($bytes) {
        $units = array('B', 'KB', 'MB', 'GB', 'TB');
        
        for ($i = 0; $bytes > 1024; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, 2) . ' ' . $units[$i];
    }
}

if (!function_exists('get_status_badge')) {
    /**
     * Get Bootstrap 5 badge HTML for status
     * 
     * @param string $status Status value
     * @return string HTML badge
     */
    function get_status_badge($status) {
        $badges = array(
            'active' => '<span class="badge bg-success">Active</span>',
            'inactive' => '<span class="badge bg-secondary">Inactive</span>',
            'pending' => '<span class="badge bg-warning text-dark">Pending</span>',
            'approved' => '<span class="badge bg-success">Approved</span>',
            'rejected' => '<span class="badge bg-danger">Rejected</span>',
            'published' => '<span class="badge bg-success">Published</span>',
            'draft' => '<span class="badge bg-secondary">Draft</span>',
            'archived' => '<span class="badge bg-dark">Archived</span>',
            'new' => '<span class="badge bg-info text-dark">New</span>',
            'contacted' => '<span class="badge bg-primary">Contacted</span>',
            'scheduled' => '<span class="badge bg-warning text-dark">Scheduled</span>',
            'completed' => '<span class="badge bg-success">Completed</span>',
            'cancelled' => '<span class="badge bg-danger">Cancelled</span>',
            'read' => '<span class="badge bg-primary">Read</span>',
            'replied' => '<span class="badge bg-success">Replied</span>',
            'upcoming' => '<span class="badge bg-info">Upcoming</span>',
            'ongoing' => '<span class="badge bg-primary">Ongoing</span>'
        );
        
        return isset($badges[$status]) ? $badges[$status] : '<span class="badge bg-light text-dark">' . ucfirst($status) . '</span>';
    }
}

if (!function_exists('is_logged_in')) {
    /**
     * Check if admin is logged in
     * 
     * @return bool Login status
     */
    function is_logged_in() {
        $CI =& get_instance();
        return $CI->session->userdata('admin_logged_in') === TRUE;
    }
}

if (!function_exists('get_admin_data')) {
    /**
     * Get logged in admin data
     * 
     * @param string $key Specific key to retrieve
     * @return mixed Admin data
     */
    function get_admin_data($key = null) {
        $CI =& get_instance();
        $admin_data = $CI->session->userdata('admin_data');
        
        if ($key) {
            return isset($admin_data[$key]) ? $admin_data[$key] : null;
        }
        
        return $admin_data;
    }
}

if (!function_exists('check_admin_permission')) {
    /**
     * Check if admin has permission for action
     * 
     * @param string $permission Permission to check
     * @return bool Permission status
     */
    function check_admin_permission($permission) {
        $role = get_admin_data('role');
        
        // Super admin has all permissions
        if ($role == 'super_admin') {
            return true;
        }
        
        // Define role permissions
        $permissions = array(
            'admin' => array('view', 'create', 'edit', 'delete'),
            'editor' => array('view', 'create', 'edit')
        );
        
        if (isset($permissions[$role]) && in_array($permission, $permissions[$role])) {
            return true;
        }
        
        return false;
    }
}

if (!function_exists('set_flash_message')) {
    /**
     * Set flash message
     * 
     * @param string $type Message type (success, error, warning, info)
     * @param string $message Message content
     */
    function set_flash_message($type, $message) {
        $CI =& get_instance();
        $CI->session->set_flashdata('message_type', $type);
        $CI->session->set_flashdata('message', $message);
    }
}

if (!function_exists('display_flash_message')) {
    /**
     * Display flash message HTML
     * 
     * @return string Flash message HTML
     */
    function display_flash_message() {
        $CI =& get_instance();
        $type = $CI->session->flashdata('message_type');
        $message = $CI->session->flashdata('message');
        
        if ($message) {
            $alert_types = array(
                'success' => 'alert-success',
                'error' => 'alert-danger',
                'warning' => 'alert-warning',
                'info' => 'alert-info'
            );
            
            $alert_class = isset($alert_types[$type]) ? $alert_types[$type] : 'alert-info';
            
            return '
            <div class="alert ' . $alert_class . ' alert-dismissible fade show" role="alert">
                ' . $message . '
                <button type="button" class="close" data-dismiss="alert" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>';
        }
        
        return '';
    }
}

if (!function_exists('generate_pagination')) {
    /**
     * Generate pagination configuration
     * 
     * @param string $base_url Base URL for pagination
     * @param int $total_rows Total number of rows
     * @param int $per_page Items per page
     * @param int $uri_segment URI segment for page number
     * @return array Pagination config
     */
    function generate_pagination($base_url, $total_rows, $per_page = 10, $uri_segment = 3) {
        $config['base_url'] = $base_url;
        $config['total_rows'] = $total_rows;
        $config['per_page'] = $per_page;
        $config['uri_segment'] = $uri_segment;
        
        // Bootstrap 4 styling
        $config['full_tag_open'] = '<nav><ul class="pagination">';
        $config['full_tag_close'] = '</ul></nav>';
        
        $config['first_link'] = 'First';
        $config['first_tag_open'] = '<li class="page-item">';
        $config['first_tag_close'] = '</li>';
        
        $config['last_link'] = 'Last';
        $config['last_tag_open'] = '<li class="page-item">';
        $config['last_tag_close'] = '</li>';
        
        $config['next_link'] = '&raquo;';
        $config['next_tag_open'] = '<li class="page-item">';
        $config['next_tag_close'] = '</li>';
        
        $config['prev_link'] = '&laquo;';
        $config['prev_tag_open'] = '<li class="page-item">';
        $config['prev_tag_close'] = '</li>';
        
        $config['cur_tag_open'] = '<li class="page-item active"><a class="page-link" href="#">';
        $config['cur_tag_close'] = '</a></li>';
        
        $config['num_tag_open'] = '<li class="page-item">';
        $config['num_tag_close'] = '</li>';
        
        $config['attributes'] = array('class' => 'page-link');
        
        return $config;
    }
}

if (!function_exists('export_to_csv')) {
    /**
     * Export data to CSV
     * 
     * @param array $data Data to export
     * @param string $filename CSV filename
     */
    function export_to_csv($data, $filename = 'export.csv') {
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        
        $output = fopen('php://output', 'w');
        
        // Add BOM for Excel UTF-8 support
        fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));
        
        // Write headers
        if (!empty($data)) {
            fputcsv($output, array_keys($data[0]));
        }
        
        // Write data
        foreach ($data as $row) {
            fputcsv($output, $row);
        }
        
        fclose($output);
        exit;
    }
}

if (!function_exists('clean_input')) {
    /**
     * Clean and sanitize input data
     * 
     * @param string $data Input data
     * @return string Cleaned data
     */
    function clean_input($data) {
        $data = trim($data);
        $data = stripslashes($data);
        $data = htmlspecialchars($data, ENT_QUOTES, 'UTF-8');
        return $data;
    }
}