<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Admin Base Controller
 * All admin controllers should extend this class
 * Location: application/core/Admin_Controller.php
 */

class Admin_Controller extends CI_Controller {

    protected $data = array();
    protected $admin_id;
    protected $admin_role;
    protected $admin_data;

    public function __construct() {
        parent::__construct();
        
        // Load required libraries and helpers
        $this->load->library('auth_library');
        $this->load->helper(array('MY_utility', 'MY_email'));
        $this->load->model(array(
            'Common_model',
            'Blog_model',
            'Volunteer_model',
            'Consultation_model',
            'Contact_model'
        ));

        // Require admin login
        $this->auth_library->require_login();

        // Get admin data
        $this->admin_id = $this->auth_library->get_admin_id();
        $this->admin_role = $this->auth_library->get_admin_role();
        $this->admin_data = $this->auth_library->get_admin_data();

        // Load common admin data
        $this->load_common_data();
    }

    /**
     * Load common data for all admin pages
     */
    protected function load_common_data() {
        // Site settings
        $this->data['site_settings'] = $this->Common_model->get_all_settings();
        
        // Admin data
        $this->data['admin_id'] = $this->admin_id;
        $this->data['admin_role'] = $this->admin_role;
        $this->data['admin_data'] = $this->admin_data;
        
        // Statistics for sidebar/dashboard
        $this->data['stats'] = array(
            'blog_posts' => $this->Blog_model->count_posts('published'),
            'draft_posts' => $this->Blog_model->count_posts('draft'),
            'volunteers_pending' => $this->Volunteer_model->count_volunteers('pending'),
            'volunteers_active' => $this->Volunteer_model->count_volunteers('active'),
            'consultations_new' => $this->Consultation_model->count_consultations('new'),
            'consultations_scheduled' => $this->Consultation_model->count_consultations('scheduled'),
            'messages_unread' => $this->Contact_model->get_unread_count()
        );
        
        // Notifications
        $this->data['notifications'] = $this->get_notifications();
        
        // Current URI for active menu
        $this->data['current_uri'] = uri_string();
        
        // Page title default
        $this->data['page_title'] = 'Admin Dashboard';
        $this->data['page_subtitle'] = '';
    }

    /**
     * Get admin notifications
     */
    protected function get_notifications() {
        $notifications = array();
        
        // New consultation requests
        if ($this->data['stats']['consultations_new'] > 0) {
            $notifications[] = array(
                'type' => 'info',
                'icon' => 'fa-calendar',
                'message' => $this->data['stats']['consultations_new'] . ' new consultation request(s)',
                'link' => 'admin/consultations'
            );
        }
        
        // Pending volunteers
        if ($this->data['stats']['volunteers_pending'] > 0) {
            $notifications[] = array(
                'type' => 'warning',
                'icon' => 'fa-users',
                'message' => $this->data['stats']['volunteers_pending'] . ' volunteer application(s) pending review',
                'link' => 'admin/volunteers'
            );
        }
        
        // Unread messages
        if ($this->data['stats']['messages_unread'] > 0) {
            $notifications[] = array(
                'type' => 'primary',
                'icon' => 'fa-envelope',
                'message' => $this->data['stats']['messages_unread'] . ' unread message(s)',
                'link' => 'admin/messages'
            );
        }
        
        return $notifications;
    }

    /**
     * Render admin view with layout
     */
    protected function render($view, $data = array()) {
        // Merge data
        $this->data = array_merge($this->data, $data);
        
        // Load view parts
        $this->data['content'] = $this->load->view('admin/' . $view, $this->data, TRUE);
        $this->load->view('admin/layouts/main', $this->data);
    }

    /**
     * Check if admin has permission
     */
    protected function check_permission($permission) {
        if (!$this->auth_library->has_permission($permission)) {
            set_flash_message('error', 'You do not have permission to perform this action.');
            redirect($_SERVER['HTTP_REFERER']);
        }
    }

    /**
     * Check if admin has role
     */
    protected function check_role($roles) {
        if (!$this->auth_library->has_role($roles)) {
            set_flash_message('error', 'You do not have permission to access this page.');
            redirect('admin/dashboard');
        }
    }

    /**
     * JSON response helper
     */
    protected function json_response($data) {
        $this->output
            ->set_content_type('application/json')
            ->set_output(json_encode($data));
    }

    /**
     * Success response
     */
    protected function success_response($message, $data = null) {
        $response = array(
            'status' => 'success',
            'message' => $message
        );
        
        if ($data) {
            $response['data'] = $data;
        }
        
        $this->json_response($response);
    }

    /**
     * Error response
     */
    protected function error_response($message, $errors = null) {
        $response = array(
            'status' => 'error',
            'message' => $message
        );
        
        if ($errors) {
            $response['errors'] = $errors;
        }
        
        $this->json_response($response);
    }

    /**
     * Upload file helper
     */
    protected function handle_upload($field_name, $upload_type = 'image', $upload_path = null) {
        $this->load->library('upload');
        
        switch ($upload_type) {
            case 'profile':
                $result = $this->upload->upload_profile_photo($field_name, $upload_path);
                break;
            case 'blog':
                $result = $this->upload->upload_blog_image($field_name, $upload_path);
                break;
            case 'document':
                $result = $this->upload->upload_document($field_name, $upload_path);
                break;
            case 'carousel':
                $result = $this->upload->upload_carousel_media($field_name, $upload_path);
                break;
            default:
                $result = $this->upload->upload_image($field_name, $upload_path);
                break;
        }
        
        return $result;
    }

    /**
     * Validate CSRF token for AJAX requests
     */
    protected function validate_csrf() {
        $csrf_token = $this->input->post('csrf_token_ngo') ?: $this->input->get('csrf_token_ngo');
        $csrf_cookie = $this->input->cookie('csrf_cookie_ngo');
        
        if ($csrf_token !== $csrf_cookie) {
            $this->error_response('Invalid security token. Please refresh the page.');
            exit;
        }
    }

    /**
     * Delete file helper
     */
    protected function delete_file($file_path) {
        if (file_exists($file_path)) {
            return unlink($file_path);
        }
        return false;
    }

    /**
     * Export to CSV helper
     */
    protected function export_csv($data, $filename) {
        if (empty($data)) {
            set_flash_message('warning', 'No data to export.');
            redirect($_SERVER['HTTP_REFERER']);
            return;
        }
        
        export_to_csv($data, $filename);
    }

    /**
     * Set page title
     */
    protected function set_title($title, $subtitle = '') {
        $this->data['page_title'] = $title;
        $this->data['page_subtitle'] = $subtitle;
    }

    /**
     * Add breadcrumb
     */
    protected function add_breadcrumb($title, $url = null) {
        if (!isset($this->data['breadcrumbs'])) {
            $this->data['breadcrumbs'] = array();
        }
        
        $this->data['breadcrumbs'][] = array(
            'title' => $title,
            'url' => $url
        );
    }

    /**
     * Log admin activity
     */
    protected function log_activity($action, $description, $affected_id = null, $affected_type = null) {
        log_message('info', sprintf(
            'Admin Activity - User: %s (ID: %d), Action: %s, Description: %s, Affected: %s (ID: %s)',
            $this->admin_data['username'],
            $this->admin_id,
            $action,
            $description,
            $affected_type,
            $affected_id
        ));
    }
}