<?php
defined('BASEPATH') OR exit('No direct script access allowed');

// Manually load Admin_Controller if not autoloaded
if (!class_exists('Admin_Controller')) {
    require_once APPPATH . 'core/Admin_Controller.php';
}


/**
 * Admin Workshops Controller
 * Handles workshop programmes management
 * Location: application/controllers/admin/Workshops.php
 */

class Workshops extends Admin_Controller {

    public function __construct() {
        parent::__construct();
    }

    /**
     * Workshops listing
     */
    public function index() {
        $this->set_title('Workshop Programmes', 'Manage Workshops');
        $this->add_breadcrumb('Dashboard', 'admin/dashboard');
        $this->add_breadcrumb('Workshops', null);

        $status = $this->input->get('status');
        $data['workshops'] = $this->Common_model->get_workshops($status);
        $data['current_status'] = $status;

		// Get statistics for all statuses
		$data['statistics'] = array(
			'total' => $this->Common_model->count_workshops(null),
			'upcoming' => $this->Common_model->count_workshops('upcoming'),
			'ongoing' => $this->Common_model->count_workshops('ongoing'),
			'completed' => $this->Common_model->count_workshops('completed'),
			'cancelled' => $this->Common_model->count_workshops('cancelled')
		);


        $this->render('workshops/index', $data);
    }

    /**
     * Add new workshop
     */
    public function add() {
        $this->check_permission('create');
        
        $this->set_title('Add Workshop', 'Create New Programme');
        $this->add_breadcrumb('Dashboard', 'admin/dashboard');
        $this->add_breadcrumb('Workshops', 'admin/workshops');
        $this->add_breadcrumb('Add New', null);

        if ($this->input->method() === 'post') {
            // Validation rules
            $this->form_validation->set_rules('title', 'Title', 'required|max_length[255]');
            $this->form_validation->set_rules('description', 'Description', 'required');
            $this->form_validation->set_rules('status', 'Status', 'required|in_list[upcoming,ongoing,completed,cancelled]');

            if ($this->form_validation->run() === TRUE) {
                $data = array(
                    'title' => $this->input->post('title', TRUE),
                    'description' => $this->input->post('description', TRUE),
                    'full_description' => $this->input->post('full_description'),
                    'duration' => $this->input->post('duration', TRUE),
                    'target_audience' => $this->input->post('target_audience', TRUE),
                    'objectives' => $this->input->post('objectives', TRUE),
                    'requirements' => $this->input->post('requirements', TRUE),
                    'schedule' => $this->input->post('schedule', TRUE),
                    'location' => $this->input->post('location', TRUE),
                    'capacity' => $this->input->post('capacity', TRUE),
                    'facilitator' => $this->input->post('facilitator', TRUE),
                    'price' => $this->input->post('price', TRUE) ?: 0.00,
                    'status' => $this->input->post('status', TRUE),
                    'start_date' => $this->input->post('start_date', TRUE) ?: NULL,
                    'end_date' => $this->input->post('end_date', TRUE) ?: NULL,
                    'display_order' => $this->input->post('display_order', TRUE) ?: 0,
                    'is_featured' => $this->input->post('is_featured') ? 1 : 0
                );

                // Handle featured image upload
                if (!empty($_FILES['featured_image']['name'])) {
                    $upload = $this->handle_upload('featured_image', 'image', './uploads/workshops/');
                    if ($upload['status']) {
                        $data['featured_image'] = $upload['data']['file_name'];
                    } else {
                        set_flash_message('error', $upload['error']);
                        redirect('admin/workshops/add');
                    }
                }

                if ($this->Common_model->create_workshop($data)) {
                    $this->log_activity('create', 'Created workshop: ' . $data['title']);
                    set_flash_message('success', 'Workshop created successfully!');
                    redirect('admin/workshops');
                } else {
                    set_flash_message('error', 'Failed to create workshop. Please try again.');
                }
            }
        }

        $this->render('workshops/form');
    }

    /**
     * Edit workshop
     */
    public function edit($id) {
        $this->check_permission('edit');
        
        $workshop = $this->Common_model->get_workshop($id);
        if (!$workshop) {
            show_404();
        }

        $this->set_title('Edit Workshop', $workshop->title);
        $this->add_breadcrumb('Dashboard', 'admin/dashboard');
        $this->add_breadcrumb('Workshops', 'admin/workshops');
        $this->add_breadcrumb('Edit', null);

        if ($this->input->method() === 'post') {
            $this->form_validation->set_rules('title', 'Title', 'required|max_length[255]');
            $this->form_validation->set_rules('description', 'Description', 'required');
            $this->form_validation->set_rules('status', 'Status', 'required|in_list[upcoming,ongoing,completed,cancelled]');

            if ($this->form_validation->run() === TRUE) {
                $data = array(
                    'title' => $this->input->post('title', TRUE),
                    'description' => $this->input->post('description', TRUE),
                    'full_description' => $this->input->post('full_description'),
                    'duration' => $this->input->post('duration', TRUE),
                    'target_audience' => $this->input->post('target_audience', TRUE),
                    'objectives' => $this->input->post('objectives', TRUE),
                    'requirements' => $this->input->post('requirements', TRUE),
                    'schedule' => $this->input->post('schedule', TRUE),
                    'location' => $this->input->post('location', TRUE),
                    'capacity' => $this->input->post('capacity', TRUE),
                    'facilitator' => $this->input->post('facilitator', TRUE),
                    'price' => $this->input->post('price', TRUE) ?: 0.00,
                    'status' => $this->input->post('status', TRUE),
                    'start_date' => $this->input->post('start_date', TRUE) ?: NULL,
                    'end_date' => $this->input->post('end_date', TRUE) ?: NULL,
                    'display_order' => $this->input->post('display_order', TRUE) ?: 0,
                    'is_featured' => $this->input->post('is_featured') ? 1 : 0
                );

                // Handle featured image upload
                if (!empty($_FILES['featured_image']['name'])) {
                    // Delete old image
                    if ($workshop->featured_image) {
                        $this->delete_file('./uploads/workshops/' . $workshop->featured_image);
                    }

                    $upload = $this->handle_upload('featured_image', 'image', './uploads/workshops/');
                    if ($upload['status']) {
                        $data['featured_image'] = $upload['data']['file_name'];
                    }
                }

                if ($this->Common_model->update_workshop($id, $data)) {
                    $this->log_activity('update', 'Updated workshop: ' . $data['title'], $id, 'workshop');
                    set_flash_message('success', 'Workshop updated successfully!');
                    redirect('admin/workshops');
                } else {
                    set_flash_message('error', 'Failed to update workshop. Please try again.');
                }
            }
        }

        $data['workshop'] = $workshop;
        $this->render('workshops/form', $data);
    }

    /**
     * Delete workshop
     */
    public function delete($id) {
        $this->check_permission('delete');
        
        $workshop = $this->Common_model->get_workshop($id);
        if (!$workshop) {
            set_flash_message('error', 'Workshop not found.');
            redirect('admin/workshops');
        }

        // Delete featured image
        if ($workshop->featured_image) {
            $this->delete_file('./uploads/workshops/' . $workshop->featured_image);
        }

        if ($this->Common_model->delete_workshop($id)) {
            $this->log_activity('delete', 'Deleted workshop: ' . $workshop->title, $id, 'workshop');
            set_flash_message('success', 'Workshop deleted successfully!');
        } else {
            set_flash_message('error', 'Failed to delete workshop.');
        }

        redirect('admin/workshops');
    }
} 