<?php
defined('BASEPATH') OR exit('No direct script access allowed');

// Manually load Admin_Controller if not autoloaded
if (!class_exists('Admin_Controller')) {
    require_once APPPATH . 'core/Admin_Controller.php';
}


/**
 * Admin Volunteers Controller
 * Handles volunteer applications management
 * Location: application/controllers/admin/Volunteers.php
 */

class Volunteers extends Admin_Controller {

    public function __construct() {
        parent::__construct();
    }

    /**
     * Volunteers listing
     */
    public function index($page = 0) {
        $this->set_title('Volunteers', 'Manage Volunteer Applications');
        $this->add_breadcrumb('Dashboard', 'admin/dashboard');
        $this->add_breadcrumb('Volunteers', null);

        // Pagination
        $per_page = 20;
        $status = $this->input->get('status');
        $total_rows = $this->Volunteer_model->count_volunteers($status);

        $this->load->library('pagination');
        $config = generate_pagination(base_url('admin/volunteers/index'), $total_rows, $per_page, 4);
        $this->pagination->initialize($config);

        $data['volunteers'] = $this->Volunteer_model->get_volunteers($per_page, $page, $status);
        $data['pagination'] = $this->pagination->create_links();
        $data['current_status'] = $status;

        // Statistics
        $data['statistics'] = $this->Volunteer_model->get_statistics();

        $this->render('volunteers/index', $data);
    }

    /**
     * View volunteer details
     */
    public function view($id) {
        $volunteer = $this->Volunteer_model->get_volunteer($id);
        
        if (!$volunteer) {
            show_404();
        }

        $this->set_title('Volunteer Details', $volunteer->full_name);
        $this->add_breadcrumb('Dashboard', 'admin/dashboard');
        $this->add_breadcrumb('Volunteers', 'admin/volunteers');
        $this->add_breadcrumb($volunteer->full_name, null);

        $data['volunteer'] = $volunteer;
        $this->render('volunteers/view', $data);
    }

    /**
     * Approve volunteer
     */
    public function approve($id) {
        $this->check_permission('edit');
        
        $volunteer = $this->Volunteer_model->get_volunteer($id);
        
        if (!$volunteer) {
            set_flash_message('error', 'Volunteer not found.');
            redirect('admin/volunteers');
        }

        if ($this->Volunteer_model->approve_volunteer($id, $this->admin_id)) {
            // Send approval email
            send_volunteer_approval($volunteer);
            
            $this->log_activity('approve', 'Approved volunteer: ' . $volunteer->full_name, $id, 'volunteer');
            set_flash_message('success', 'Volunteer approved successfully! Approval email sent.');
        } else {
            set_flash_message('error', 'Failed to approve volunteer.');
        }

        redirect('admin/volunteers/view/' . $id);
    }

    /**
     * Reject volunteer
     */
    public function reject($id) {
        $this->check_permission('edit');
        
        $volunteer = $this->Volunteer_model->get_volunteer($id);
        
        if (!$volunteer) {
            set_flash_message('error', 'Volunteer not found.');
            redirect('admin/volunteers');
        }

        if ($this->Volunteer_model->reject_volunteer($id, $this->admin_id)) {
            $this->log_activity('reject', 'Rejected volunteer: ' . $volunteer->full_name, $id, 'volunteer');
            set_flash_message('success', 'Volunteer application rejected.');
        } else {
            set_flash_message('error', 'Failed to reject volunteer.');
        }

        redirect('admin/volunteers/view/' . $id);
    }

    /**
     * Activate volunteer (make visible on website)
     */
    public function activate($id) {
        $this->check_permission('edit');
        
        if ($this->Volunteer_model->activate_volunteer($id)) {
            set_flash_message('success', 'Volunteer activated and now visible on website.');
        } else {
            set_flash_message('error', 'Failed to activate volunteer.');
        }

        redirect('admin/volunteers/view/' . $id);
    }

    /**
     * Update volunteer notes
     */
    public function update_notes($id) {
        if ($this->input->method() === 'post') {
            $notes = $this->input->post('notes', TRUE);
            
            if ($this->Volunteer_model->add_notes($id, $notes)) {
                set_flash_message('success', 'Notes updated successfully.');
            } else {
                set_flash_message('error', 'Failed to update notes.');
            }
        }

        redirect('admin/volunteers/view/' . $id);
    }

    /**
     * Delete volunteer
     */
    public function delete($id) {
        $this->check_permission('delete');
        
        $volunteer = $this->Volunteer_model->get_volunteer($id);
        
        if (!$volunteer) {
            set_flash_message('error', 'Volunteer not found.');
            redirect('admin/volunteers');
        }

        // Delete profile photo
        if ($volunteer->profile_photo) {
            $this->delete_file('./uploads/volunteers/' . $volunteer->profile_photo);
        }

        if ($this->Volunteer_model->delete_volunteer($id)) {
            $this->log_activity('delete', 'Deleted volunteer: ' . $volunteer->full_name, $id, 'volunteer');
            set_flash_message('success', 'Volunteer deleted successfully.');
        } else {
            set_flash_message('error', 'Failed to delete volunteer.');
        }

        redirect('admin/volunteers');
    }

    /**
     * Export volunteers to CSV
     */
    public function export() {
        $status = $this->input->get('status');
        $volunteers = $this->Volunteer_model->get_volunteers_for_export($status);
        
        $filename = 'volunteers_' . date('Y-m-d') . '.csv';
        $this->export_csv($volunteers, $filename);
    }

    /**
     * Search volunteers
     */
    public function search() {
        $keyword = $this->input->get('keyword');
        $status = $this->input->get('status');

        if (empty($keyword)) {
            redirect('admin/volunteers');
        }

        $this->set_title('Search Results', 'Volunteers matching "' . $keyword . '"');
        $this->add_breadcrumb('Dashboard', 'admin/dashboard');
        $this->add_breadcrumb('Volunteers', 'admin/volunteers');
        $this->add_breadcrumb('Search Results', null);

        $data['volunteers'] = $this->Volunteer_model->search_volunteers($keyword, $status, 50, 0);
        $data['search_keyword'] = $keyword;
        $data['current_status'] = $status;
        $data['statistics'] = $this->Volunteer_model->get_statistics();

        $this->render('volunteers/index', $data);
    }
}