<?php
defined('BASEPATH') OR exit('No direct script access allowed');

// Manually load Admin_Controller if not autoloaded
if (!class_exists('Admin_Controller')) {
    require_once APPPATH . 'core/Admin_Controller.php';
}

/**
 * Admin Partners Controller
 * Handles partners/sponsors management
 * Location: application/controllers/admin/Partners.php
 */

class Partners extends Admin_Controller {

    public function __construct() {
        parent::__construct();
    }

    /**
     * Partners listing
     */
    public function index() {
        $this->set_title('Partners', 'Manage Partners & Sponsors');
        $this->add_breadcrumb('Dashboard', 'admin/dashboard');
        $this->add_breadcrumb('Partners', null);

        $data['partners'] = $this->Common_model->get_partners();

        $this->render('partners/index', $data);
    }

    /**
     * Add new partner
     */
    public function add() {
        $this->check_permission('create');
        
        $this->set_title('Add Partner', 'Add New Partner/Sponsor');
        $this->add_breadcrumb('Dashboard', 'admin/dashboard');
        $this->add_breadcrumb('Partners', 'admin/partners');
        $this->add_breadcrumb('Add New', null);

        if ($this->input->method() === 'post') {
            $this->form_validation->set_rules('name', 'Partner Name', 'required|max_length[200]');
            $this->form_validation->set_rules('partnership_type', 'Partnership Type', 'required|in_list[corporate,ngo,government,academic,individual]');

            if ($this->form_validation->run() === TRUE) {
                $data = array(
                    'name' => $this->input->post('name', TRUE),
                    'website' => $this->input->post('website', TRUE),
                    'description' => $this->input->post('description', TRUE),
                    'partnership_type' => $this->input->post('partnership_type', TRUE),
                    'display_order' => $this->input->post('display_order', TRUE) ?: 0,
                    'status' => $this->input->post('status', TRUE) ?: 'active'
                );

                // Handle logo upload
                if (!empty($_FILES['logo']['name'])) {
                    $upload = $this->handle_upload('logo', 'image', './uploads/partners/');
                    if ($upload['status']) {
                        $data['logo'] = $upload['data']['file_name'];
                    } else {
                        set_flash_message('error', $upload['error']);
                        redirect('admin/partners/add');
                    }
                }

                if ($this->Common_model->create_partner($data)) {
                    $this->log_activity('create', 'Added partner: ' . $data['name']);
                    set_flash_message('success', 'Partner added successfully!');
                    redirect('admin/partners');
                } else {
                    set_flash_message('error', 'Failed to add partner. Please try again.');
                }
            }
        }

        $this->render('partners/form');
    }

    /**
     * Edit partner
     */
    public function edit($id) {
        $this->check_permission('edit');
        
        $partner = $this->Common_model->get_partner($id);
        if (!$partner) {
            show_404();
        }

        $this->set_title('Edit Partner', $partner->name);
        $this->add_breadcrumb('Dashboard', 'admin/dashboard');
        $this->add_breadcrumb('Partners', 'admin/partners');
        $this->add_breadcrumb('Edit', null);

        if ($this->input->method() === 'post') {
            $this->form_validation->set_rules('name', 'Partner Name', 'required|max_length[200]');
            $this->form_validation->set_rules('partnership_type', 'Partnership Type', 'required|in_list[corporate,ngo,government,academic,individual]');

            if ($this->form_validation->run() === TRUE) {
			$data = array(
				'name' => $this->input->post('name', TRUE),
				'website' => $this->input->post('website', TRUE),
				'description' => $this->input->post('description', TRUE),
				'partnership_type' => $this->input->post('partnership_type', TRUE),
				'display_order' => $this->input->post('display_order', TRUE) ?: 0,
				'status' => $this->input->post('status', TRUE)
			);

			// Handle logo upload - ONLY if new file uploaded
			if (!empty($_FILES['logo']['name'])) {
				// Delete old logo
				if ($partner->logo) {
					$this->delete_file('./uploads/partners/' . $partner->logo);
				}

				$upload = $this->handle_upload('logo', 'image', './uploads/partners/');
				if ($upload['status']) {
					$data['logo'] = $upload['data']['file_name'];
				} else {
					set_flash_message('error', $upload['error']);
					redirect('admin/partners/edit/' . $id);
				}
			}
			// If no new file, don't update logo field - it stays as is

                if ($this->Common_model->update_partner($id, $data)) {
                    $this->log_activity('update', 'Updated partner: ' . $data['name'], $id, 'partner');
                    set_flash_message('success', 'Partner updated successfully!');
                    redirect('admin/partners');
                } else {
                    set_flash_message('error', 'Failed to update partner. Please try again.');
                }
            }
        }

        $data['partner'] = $partner;
        $this->render('partners/form', $data);
    }

    /**
     * Delete partner
     */
    public function delete($id) {
        $this->check_permission('delete');
        
        $partner = $this->Common_model->get_partner($id);
        if (!$partner) {
            set_flash_message('error', 'Partner not found.');
            redirect('admin/partners');
        }

        // Delete logo
        if ($partner->logo) {
            $this->delete_file('./uploads/partners/' . $partner->logo);
        }

        if ($this->Common_model->delete_partner($id)) {
            $this->log_activity('delete', 'Deleted partner: ' . $partner->name, $id, 'partner');
            set_flash_message('success', 'Partner deleted successfully!');
        } else {
            set_flash_message('error', 'Failed to delete partner.');
        }

        redirect('admin/partners');
    }
}