<?php
defined('BASEPATH') OR exit('No direct script access allowed');

// Manually load Admin_Controller if not autoloaded
if (!class_exists('Admin_Controller')) {
    require_once APPPATH . 'core/Admin_Controller.php';
}

/**
 * Admin Messages Controller
 * Handles contact messages management
 * Location: application/controllers/admin/Messages.php
 */

class Messages extends Admin_Controller {

    public function __construct() {
        parent::__construct();
    }

    /**
     * Messages listing
     */
    public function index($page = 0) {
        $this->set_title('Contact Messages', 'Manage Inquiries');
        $this->add_breadcrumb('Dashboard', 'admin/dashboard');
        $this->add_breadcrumb('Messages', null);

        // Pagination
        $per_page = 20;
        $status = $this->input->get('status');
        $total_rows = $this->Contact_model->count_messages($status);

        $this->load->library('pagination');
        $config = generate_pagination(base_url('admin/messages/index'), $total_rows, $per_page, 4);
        $this->pagination->initialize($config);

        $data['messages'] = $this->Contact_model->get_messages($per_page, $page, $status);
        $data['pagination'] = $this->pagination->create_links();
        $data['current_status'] = $status;

        // Statistics
        $data['statistics'] = $this->Contact_model->get_statistics();
	

		// Get statistics for all statuses
		$data['statistics'] = array(
			'total' => $this->Contact_model->count_messages(null),
			'new' => $this->Contact_model->count_messages('new'),
			'read' => $this->Contact_model->count_messages('read'),
			'replied' => $this->Contact_model->count_messages('replied'),
			'archived' => $this->Contact_model->count_messages('archived')
		);

        $this->render('messages/index', $data);
    }

    /**
     * View message details
     */
    public function view($id) {
        $message = $this->Contact_model->get_message($id);
        
        if (!$message) {
            show_404();
        }

        // Mark as read if status is 'new'
        if ($message->status === 'new') {
            $this->Contact_model->mark_as_read($id);
        }

        $this->set_title('Message Details', $message->name);
        $this->add_breadcrumb('Dashboard', 'admin/dashboard');
        $this->add_breadcrumb('Messages', 'admin/messages');
        $this->add_breadcrumb($message->name, null);

        $data['message'] = $message;
        $this->render('messages/view', $data);
    }

    /**
     * Reply to message
     */
    public function reply($id) {
        $this->check_permission('edit');
        
        $message = $this->Contact_model->get_message($id);
        
        if (!$message) {
            show_404();
        }

        $this->set_title('Reply to Message', $message->name);
        $this->add_breadcrumb('Dashboard', 'admin/dashboard');
        $this->add_breadcrumb('Messages', 'admin/messages');
        $this->add_breadcrumb('Reply', null);

        if ($this->input->method() === 'post') {
            $this->form_validation->set_rules('reply_message', 'Reply Message', 'required|min_length[10]');

            if ($this->form_validation->run() === TRUE) {
                $reply_message = $this->input->post('reply_message', TRUE);
                
                // Update message status
                if ($this->Contact_model->mark_as_replied($id, $this->admin_id, $reply_message)) {
                    // Send reply email
                    $admin_name = $this->admin_data['full_name'];
                    $site_name = $this->data['site_settings']['site_name'];
                    
                    $subject = "Re: {$message->subject}";
                    $email_body = "
                    <h3>Response to Your Message</h3>
                    <p>Dear {$message->name},</p>
                    <p>Thank you for contacting us. Here is our response:</p>
                    <div style='background:#f9f9f9;padding:15px;margin:20px 0;'>
                        " . nl2br($reply_message) . "
                    </div>
                    <p>Your original message:</p>
                    <div style='background:#f0f0f0;padding:15px;margin:20px 0;'>
                        <strong>Subject:</strong> {$message->subject}<br>
                        " . nl2br($message->message) . "
                    </div>
                    <p>If you have any further questions, please don't hesitate to contact us again.</p>
                    <p>Best regards,<br>$admin_name<br>$site_name</p>
                    ";
                    
                    send_email($message->email, $subject, $email_body);
                    
                    $this->log_activity('reply', 'Replied to message from: ' . $message->name, $id, 'contact_message');
                    set_flash_message('success', 'Reply sent successfully!');
                    redirect('admin/messages/view/' . $id);
                } else {
                    set_flash_message('error', 'Failed to send reply.');
                }
            }
        }

        $data['message'] = $message;
        $this->render('messages/reply', $data);
    }

    /**
     * Archive message
     */
    public function archive($id) {
        $this->check_permission('edit');
        
        if ($this->Contact_model->archive_message($id)) {
            set_flash_message('success', 'Message archived successfully.');
        } else {
            set_flash_message('error', 'Failed to archive message.');
        }

        redirect('admin/messages');
    }

    /**
     * Delete message
     */
    public function delete($id) {
        $this->check_permission('delete');
        
        $message = $this->Contact_model->get_message($id);
        
        if (!$message) {
            set_flash_message('error', 'Message not found.');
            redirect('admin/messages');
        }

        if ($this->Contact_model->delete_message($id)) {
            $this->log_activity('delete', 'Deleted message from: ' . $message->name, $id, 'contact_message');
            set_flash_message('success', 'Message deleted successfully.');
        } else {
            set_flash_message('error', 'Failed to delete message.');
        }

        redirect('admin/messages');
    }

    /**
     * Bulk actions
     */
    public function bulk_action() {
        $this->check_permission('edit');
        
        if ($this->input->method() === 'post') {
            $action = $this->input->post('bulk_action');
            $message_ids = $this->input->post('message_ids');

            if (empty($message_ids)) {
                set_flash_message('warning', 'No messages selected.');
                redirect('admin/messages');
            }

            switch ($action) {
                case 'mark_read':
                    if ($this->Contact_model->bulk_update_status($message_ids, 'read')) {
                        set_flash_message('success', count($message_ids) . ' message(s) marked as read.');
                    }
                    break;

                case 'archive':
                    if ($this->Contact_model->bulk_update_status($message_ids, 'archived')) {
                        set_flash_message('success', count($message_ids) . ' message(s) archived.');
                    }
                    break;

                case 'delete':
                    if ($this->Contact_model->bulk_delete($message_ids)) {
                        set_flash_message('success', count($message_ids) . ' message(s) deleted.');
                    }
                    break;

                default:
                    set_flash_message('error', 'Invalid bulk action.');
            }
        }

        redirect('admin/messages');
    }

    /**
     * Export messages to CSV
     */
    public function export() {
        $status = $this->input->get('status');
        $messages = $this->Contact_model->get_messages_for_export($status);
        
        $filename = 'contact_messages_' . date('Y-m-d') . '.csv';
        $this->export_csv($messages, $filename);
    }

    /**
     * Search messages
     */
    public function search() {
        $keyword = $this->input->get('keyword');
        $status = $this->input->get('status');

        if (empty($keyword)) {
            redirect('admin/messages');
        }

        $this->set_title('Search Results', 'Messages matching "' . $keyword . '"');
        $this->add_breadcrumb('Dashboard', 'admin/dashboard');
        $this->add_breadcrumb('Messages', 'admin/messages');
        $this->add_breadcrumb('Search Results', null);

        $data['messages'] = $this->Contact_model->search_messages($keyword, $status, 50, 0);
        $data['search_keyword'] = $keyword;
        $data['current_status'] = $status;
        $data['statistics'] = $this->Contact_model->get_statistics();

        $this->render('messages/index', $data);
    }
}