<?php
defined('BASEPATH') OR exit('No direct script access allowed');

// Manually load Admin_Controller if not autoloaded
if (!class_exists('Admin_Controller')) {
    require_once APPPATH . 'core/Admin_Controller.php';
}

class Media_management extends Admin_Controller {
    public function __construct() {
        parent::__construct();
        $this->load->model(['Media_model', 'Media_category_model']);
        $this->load->helper(['url', 'form']);
        $this->load->library('upload');
    }

    /**
     * Media listing in admin panel
     */
    public function index() {
        // Set page title and metadata
        $this->set_title('Media Management', 'Upload and Manage Media');
        
        // Add breadcrumbs
        $this->add_breadcrumb('Dashboard', site_url('admin/dashboard'));
        $this->add_breadcrumb('Media Management', null);

        // Get filter parameters
        $category_id = $this->input->get('category');
        $media_type = $this->input->get('type');
        $status = $this->input->get('status');

        // Prepare data
        $data = [
            'media_items' => $this->Media_model->get_media($category_id, $media_type, $status),
            'categories' => $this->Media_category_model->get_all_categories(),
            'selected_category' => $category_id,
            'selected_type' => $media_type,
            'selected_status' => $status
        ];

        // Render view
        $this->render('media/index', $data);
    }

    /**
     * Media upload form
     */
    public function upload() {
        // Set page title and metadata
        $this->set_title('Upload Media', 'Add New Media Item');
        
        // Add breadcrumbs
        $this->add_breadcrumb('Dashboard', site_url('admin/dashboard'));
        $this->add_breadcrumb('Media Management', site_url('admin/media_management'));
        $this->add_breadcrumb('Upload Media', null);

        // Prepare data
        $data = [
            'categories' => $this->Media_category_model->get_all_categories()
        ];

        // Render view
        $this->render('media/upload', $data);
    }

    /**
     * Process media upload
     */
   public function do_upload() {
    // Validation rules (remain the same)
    $this->form_validation->set_rules('title', 'Title', 'required|max_length[255]');
    $this->form_validation->set_rules('category_id', 'Category', 'required|numeric');

    // Run validation
    if ($this->form_validation->run() === FALSE) {
        $this->session->set_flashdata('error', validation_errors());
        redirect('admin/media_management/upload');
        return;
    }

    // Determine upload subfolder based on file type
    $file_type = $this->_get_file_type($_FILES['media_file']['type']);
    $upload_dir = FCPATH . 'uploads/media/' . $file_type . 's/';

    // Ensure upload directory exists and is writable
    if (!is_dir($upload_dir)) {
        mkdir($upload_dir, 0755, TRUE);
    }

    // Upload configuration
    $config = [
        'upload_path' => $upload_dir,
        'allowed_types' => $this->_get_allowed_types($file_type),
        'max_size' => 70240, // 10MB
        'encrypt_name' => TRUE,
        'remove_spaces' => TRUE
    ];

    // Initialize upload library
    $this->upload->initialize($config);

    // Attempt file upload
    if (!$this->upload->do_upload('media_file')) {
        // Upload failed
        $error = $this->upload->display_errors();
        
        // Log detailed error
        log_message('error', 'Media upload error: ' . $error);
        
        // Set flash message for user
        $this->session->set_flashdata('error', 'Media upload failed: ' . $error);
        redirect('admin/media_management/upload');
        return;
    }

    // Upload successful
    $upload_data = $this->upload->data();

    // Prepare media data for database
    $media_data = [
        'title' => $this->input->post('title'),
        'description' => $this->input->post('description'),
        'file_path' => $file_type . 's/' . $upload_data['file_name'], // Store relative path
        'file_type' => $file_type,
        'category_id' => $this->input->post('category_id'),
        'tags' => $this->input->post('tags'),
        'status' => $this->input->post('status') ?: 'draft',
        'uploaded_by' => $this->session->userdata('user_id')
    ];

    // Save to database
    $result = $this->Media_model->insert_media($media_data);

    if ($result) {
        $this->session->set_flashdata('success', 'Media uploaded successfully');
    } else {
        // If database save fails, delete the uploaded file
        $file_path = $upload_dir . $upload_data['file_name'];
        if (file_exists($file_path)) {
            unlink($file_path);
        }
        $this->session->set_flashdata('error', 'Failed to save media information');
    }

    redirect('admin/media_management');
}

/**
 * Get allowed file types based on media type
 * @param string $type Media type (image/video/document)
 * @return string Allowed file extensions
 */
private function _get_allowed_types($type) {
    switch ($type) {
        case 'image':
            return 'gif|jpg|png|jpeg|webp';
        case 'video':
            return 'mp4|avi|mov|wmv|mkv';
        case 'document':
            return 'pdf|doc|docx|txt|csv|xls|xlsx';
        default:
            return '';
    }
}
    /**
     * Edit media item
     * @param int $id Media item ID
     */
    public function edit($id) {
        // Fetch media item
        $media_item = $this->Media_model->get_media_by_id($id);

        if (empty($media_item)) {
            show_404();
        }

        // Set page title and metadata
        $this->set_title('Edit Media', $media_item->title);
        
        // Add breadcrumbs
        $this->add_breadcrumb('Dashboard', site_url('admin/dashboard'));
        $this->add_breadcrumb('Media Management', site_url('admin/media_management'));
        $this->add_breadcrumb('Edit Media', null);

        // Prepare data
        $data = [
            'media_item' => $media_item,
            'categories' => $this->Media_category_model->get_all_categories()
        ];

        // Render view
        $this->render('media/edit', $data);
    }

    /**
     * Update media item
     */
    public function update() {
        // Validate form
        $this->form_validation->set_rules('id', 'Media ID', 'required|numeric');
        $this->form_validation->set_rules('title', 'Title', 'required|max_length[255]');
        $this->form_validation->set_rules('category_id', 'Category', 'required|numeric');

        if ($this->form_validation->run() === FALSE) {
            // Validation failed, return to edit form
            $this->session->set_flashdata('error', validation_errors());
            redirect('admin/media_management');
            return;
        }

        // Prepare update data
        $media_data = [
            'title' => $this->input->post('title'),
            'description' => $this->input->post('description'),
            'category_id' => $this->input->post('category_id'),
            'tags' => $this->input->post('tags'),
            'status' => $this->input->post('status') ?: 'draft'
        ];

        // Handle file upload
        $upload_dir = FCPATH . 'uploads/media/';
        $config = [
            'upload_path' => $upload_dir,
            'allowed_types' => 'gif|jpg|png|jpeg|mp4|pdf|doc|docx',
            'max_size' => 10240,
            'encrypt_name' => TRUE
        ];

        // Optional: Handle new file upload
        if (!empty($_FILES['media_file']['name'])) {
            $this->upload->initialize($config);

            if ($this->upload->do_upload('media_file')) {
                $upload_data = $this->upload->data();
                $media_data['file_path'] = $upload_data['file_name'];
                $media_data['file_type'] = $this->_get_file_type($upload_data['file_type']);
            } else {
                $error = $this->upload->display_errors();
                $this->session->set_flashdata('error', 'File upload failed: ' . $error);
                redirect('admin/media_management/edit/' . $this->input->post('id'));
                return;
            }
        }

        // Update media
        $result = $this->Media_model->update_media($this->input->post('id'), $media_data);

        if ($result) {
            $this->session->set_flashdata('success', 'Media updated successfully');
        } else {
            $this->session->set_flashdata('error', 'Failed to update media');
        }

        redirect('admin/media_management');
    }

    /**
     * Delete media item
     * @param int $id Media item ID
     */
    public function delete($id) {
        $result = $this->Media_model->delete_media($id);

        if ($result) {
            $this->session->set_flashdata('success', 'Media deleted successfully');
        } else {
            $this->session->set_flashdata('error', 'Failed to delete media');
        }

        redirect('admin/media_management');
    }

    /**
     * Determine file type
     * @param string $mime_type File MIME type
     * @return string File type (image/video/document)
     */
    private function _get_file_type($mime_type) {
        $image_types = ['image/jpeg', 'image/png', 'image/gif'];
        $video_types = ['video/mp4', 'video/mpeg', 'video/quicktime'];
        $document_types = ['application/pdf', 'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'];

        if (in_array($mime_type, $image_types)) return 'image';
        if (in_array($mime_type, $video_types)) return 'video';
        if (in_array($mime_type, $document_types)) return 'document';

        return 'unknown';
    }
}