<?php
defined('BASEPATH') OR exit('No direct script access allowed');

// Manually load Admin_Controller if not autoloaded
if (!class_exists('Admin_Controller')) {
    require_once APPPATH . 'core/Admin_Controller.php';
}
/**
 * Admin Export Controller
 * Handles data export functionality (CSV, Excel, etc.)
 * Location: application/controllers/admin/Export.php
 */

class Export extends Admin_Controller {

    public function __construct() {
        parent::__construct();
        $this->check_permission('view'); // Must have view permission to export
    }

    /**
     * Export volunteers to CSV
     */
    public function volunteers() {
        $status = $this->input->get('status');
        $volunteers = $this->Volunteer_model->get_volunteers_for_export($status);
        
        if (empty($volunteers)) {
            set_flash_message('warning', 'No volunteers data to export.');
            redirect('admin/volunteers');
            return;
        }

        // Prepare data for CSV
        $csv_data = array();
        foreach ($volunteers as $volunteer) {
            $csv_data[] = array(
                'ID' => $volunteer['id'],
                'Full Name' => $volunteer['full_name'],
                'Email' => $volunteer['email'],
                'Phone' => $volunteer['phone'],
                'City' => $volunteer['city'],
                'Country' => $volunteer['country'],
                'Occupation' => $volunteer['occupation'],
                'Skills' => $volunteer['skills'],
                'Availability' => ucfirst($volunteer['availability']),
                'Status' => ucfirst($volunteer['status']),
                'Application Date' => $volunteer['application_date'],
                'Approved By' => $volunteer['approved_by_name'] ?? 'N/A',
                'Approved At' => $volunteer['approved_at'] ?? 'N/A'
            );
        }

        $filename = 'volunteers_export_' . date('Y-m-d_His') . '.csv';
        $this->export_csv($csv_data, $filename);
        
        $this->log_activity('export', 'Exported volunteers data to CSV');
    }

    /**
     * Export consultations to CSV
     */
    public function consultations() {
        $status = $this->input->get('status');
        $consultations = $this->Consultation_model->get_consultations_for_export($status);
        
        if (empty($consultations)) {
            set_flash_message('warning', 'No consultations data to export.');
            redirect('admin/consultations');
            return;
        }

        // Prepare data for CSV
        $csv_data = array();
        foreach ($consultations as $consultation) {
            $csv_data[] = array(
                'ID' => $consultation['id'],
                'Full Name' => $consultation['full_name'],
                'Email' => $consultation['email'],
                'Phone' => $consultation['phone'],
                'Organization' => $consultation['organization'] ?? 'N/A',
                'Type' => ucfirst($consultation['consultation_type']),
                'Subject' => $consultation['subject'],
                'Message' => $consultation['message'],
                'Preferred Date' => $consultation['preferred_date'] ?? 'N/A',
                'Preferred Time' => $consultation['preferred_time'] ?? 'N/A',
                'Status' => ucfirst($consultation['status']),
                'Assigned To' => $consultation['assigned_to_name'] ?? 'N/A',
                'Scheduled Date' => $consultation['scheduled_date'] ?? 'N/A',
                'Created At' => $consultation['created_at']
            );
        }

        $filename = 'consultations_export_' . date('Y-m-d_His') . '.csv';
        $this->export_csv($csv_data, $filename);
        
        $this->log_activity('export', 'Exported consultations data to CSV');
    }

    /**
     * Export contact messages to CSV
     */
    public function messages() {
        $status = $this->input->get('status');
        $messages = $this->Contact_model->get_messages_for_export($status);
        
        if (empty($messages)) {
            set_flash_message('warning', 'No messages data to export.');
            redirect('admin/messages');
            return;
        }

        // Prepare data for CSV
        $csv_data = array();
        foreach ($messages as $message) {
            $csv_data[] = array(
                'ID' => $message['id'],
                'Name' => $message['name'],
                'Email' => $message['email'],
                'Phone' => $message['phone'] ?? 'N/A',
                'Subject' => $message['subject'],
                'Message' => $message['message'],
                'Status' => ucfirst($message['status']),
                'Replied By' => $message['replied_by_name'] ?? 'N/A',
                'Replied At' => $message['replied_at'] ?? 'N/A',
                'Created At' => $message['created_at'],
                'IP Address' => $message['ip_address'] ?? 'N/A'
            );
        }

        $filename = 'contact_messages_export_' . date('Y-m-d_His') . '.csv';
        $this->export_csv($csv_data, $filename);
        
        $this->log_activity('export', 'Exported contact messages data to CSV');
    }

    /**
     * Export blog posts to CSV
     */
    public function blog_posts() {
        $status = $this->input->get('status');
        
        $this->db->select('blog_posts.*, blog_categories.name as category_name, admin_users.full_name as author_name');
        $this->db->from('blog_posts');
        $this->db->join('blog_categories', 'blog_categories.id = blog_posts.category_id', 'left');
        $this->db->join('admin_users', 'admin_users.id = blog_posts.author_id', 'left');
        
        if ($status) {
            $this->db->where('blog_posts.status', $status);
        }
        
        $query = $this->db->get();
        $posts = $query->result_array();
        
        if (empty($posts)) {
            set_flash_message('warning', 'No blog posts data to export.');
            redirect('admin/blog');
            return;
        }

        // Prepare data for CSV
        $csv_data = array();
        foreach ($posts as $post) {
            $csv_data[] = array(
                'ID' => $post['id'],
                'Title' => $post['title'],
                'Category' => $post['category_name'] ?? 'Uncategorized',
                'Author' => $post['author_name'],
                'Status' => ucfirst($post['status']),
                'Views' => $post['views_count'],
                'Published At' => $post['published_at'] ?? 'N/A',
                'Created At' => $post['created_at']
            );
        }

        $filename = 'blog_posts_export_' . date('Y-m-d_His') . '.csv';
        $this->export_csv($csv_data, $filename);
        
        $this->log_activity('export', 'Exported blog posts data to CSV');
    }

    /**
     * Export all data (comprehensive export)
     */
    public function all_data() {
        $this->check_role('super_admin'); // Only super admin can export all data
        
        // This would create a comprehensive ZIP file with all exports
        // For simplicity, we'll redirect to dashboard with a message
        set_flash_message('info', 'Comprehensive data export feature coming soon. Please export individual modules for now.');
        redirect('admin/dashboard');
        
        $this->log_activity('export', 'Attempted comprehensive data export');
    }
}