<?php
defined('BASEPATH') OR exit('No direct script access allowed');

// Manually load Admin_Controller if not autoloaded
if (!class_exists('Admin_Controller')) {
    require_once APPPATH . 'core/Admin_Controller.php';
}
/**
 * Admin Consultations Controller
 * Handles consultation requests management
 * Location: application/controllers/admin/Consultations.php
 */

class Consultations extends Admin_Controller {

    public function __construct() {
        parent::__construct();
    }

    /**
     * Consultations listing
     */
    public function index($page = 0) {
        $this->set_title('Consultation Requests', 'Manage Consultation Bookings');
        $this->add_breadcrumb('Dashboard', 'admin/dashboard');
        $this->add_breadcrumb('Consultations', null);

        // Pagination
        $per_page = 20;
        $status = $this->input->get('status');
        $total_rows = $this->Consultation_model->count_consultations($status);

        $this->load->library('pagination');
        $config = generate_pagination(base_url('admin/consultations/index'), $total_rows, $per_page, 4);
        $this->pagination->initialize($config);

        $data['consultations'] = $this->Consultation_model->get_consultations($per_page, $page, $status);
        $data['pagination'] = $this->pagination->create_links();
        $data['current_status'] = $status;

        // Statistics
        $data['statistics'] = $this->Consultation_model->get_statistics();

        $this->render('consultations/index', $data);
    }

    /**
     * View consultation details
     */
    public function view($id) {
        $consultation = $this->Consultation_model->get_consultation($id);
        
        if (!$consultation) {
            show_404();
        }

        $this->set_title('Consultation Details', $consultation->full_name);
        $this->add_breadcrumb('Dashboard', 'admin/dashboard');
        $this->add_breadcrumb('Consultations', 'admin/consultations');
        $this->add_breadcrumb($consultation->full_name, null);

        // Get all admins for assignment dropdown
        $data['consultation'] = $consultation;
        $data['admins'] = $this->Common_model->get_all_admins();

        $this->render('consultations/view', $data);
    }

    /**
     * Update consultation status
     */
    public function update($id) {
        // TEMPORARILY DISABLED FOR TESTING
        // $this->check_permission('edit');
        
        if ($this->input->method() === 'post') {
            $action = $this->input->post('action');
            
            switch ($action) {
                case 'update_status':
                    $status = $this->input->post('status', TRUE);
                    $notes = $this->input->post('notes', TRUE);
                    
                    $data = array('notes' => $notes);
                    
                    if ($this->Consultation_model->update_status($id, $status, $data)) {
                        set_flash_message('success', 'Status updated successfully.');
                    } else {
                        set_flash_message('error', 'Failed to update status.');
                    }
                    break;

                case 'assign':
                    $admin_id = $this->input->post('assigned_to', TRUE);
                    
                    if ($this->Consultation_model->assign_consultation($id, $admin_id)) {
                        $this->log_activity('assign', 'Assigned consultation to admin', $id, 'consultation');
                        set_flash_message('success', 'Consultation assigned successfully.');
                    } else {
                        set_flash_message('error', 'Failed to assign consultation.');
                    }
                    break;

                case 'schedule':
                    $this->form_validation->set_rules('scheduled_date', 'Scheduled Date', 'required');
                    
                    if ($this->form_validation->run() === TRUE) {
                        $scheduled_date = $this->input->post('scheduled_date', TRUE);
                        $assigned_to = $this->input->post('assigned_to', TRUE);
                        
                        if ($this->Consultation_model->schedule_consultation($id, $scheduled_date, $assigned_to)) {
                            // Send email notification
                            $consultation = $this->Consultation_model->get_consultation($id);
                            
                            $subject = 'Consultation Scheduled';
                            $message = "
                            <h3>Your Consultation Has Been Scheduled</h3>
                            <p>Dear {$consultation->full_name},</p>
                            <p>Your consultation has been scheduled for: <strong>" . format_datetime($scheduled_date) . "</strong></p>
                            <p>We look forward to speaking with you!</p>
                            ";
                            send_email($consultation->email, $subject, $message);
                            
                            set_flash_message('success', 'Consultation scheduled successfully. Email notification sent.');
                        } else {
                            set_flash_message('error', 'Failed to schedule consultation.');
                        }
                    }
                    break;

                case 'update_notes':
                    $notes = $this->input->post('notes', TRUE);
                    
                    if ($this->Consultation_model->add_notes($id, $notes)) {
                        set_flash_message('success', 'Notes updated successfully.');
                    } else {
                        set_flash_message('error', 'Failed to update notes.');
                    }
                    break;
            }
        }

        redirect('admin/consultations/view/' . $id);
    }

    /**
     * Delete consultation
     */
    public function delete($id) {
        $this->check_permission('delete');
        
        $consultation = $this->Consultation_model->get_consultation($id);
        
        if (!$consultation) {
            set_flash_message('error', 'Consultation not found.');
            redirect('admin/consultations');
        }

        if ($this->Consultation_model->delete_consultation($id)) {
            $this->log_activity('delete', 'Deleted consultation from: ' . $consultation->full_name, $id, 'consultation');
            set_flash_message('success', 'Consultation deleted successfully.');
        } else {
            set_flash_message('error', 'Failed to delete consultation.');
        }

        redirect('admin/consultations');
    }

    /**
     * Export consultations to CSV
     */
    public function export() {
        $status = $this->input->get('status');
        $consultations = $this->Consultation_model->get_consultations_for_export($status);
        
        $filename = 'consultations_' . date('Y-m-d') . '.csv';
        $this->export_csv($consultations, $filename);
    }

    /**
     * Search consultations
     */
    public function search() {
        $keyword = $this->input->get('keyword');
        $status = $this->input->get('status');

        if (empty($keyword)) {
            redirect('admin/consultations');
        }

        $this->set_title('Search Results', 'Consultations matching "' . $keyword . '"');
        $this->add_breadcrumb('Dashboard', 'admin/dashboard');
        $this->add_breadcrumb('Consultations', 'admin/consultations');
        $this->add_breadcrumb('Search Results', null);

        $data['consultations'] = $this->Consultation_model->search_consultations($keyword, $status, 50, 0);
        $data['search_keyword'] = $keyword;
        $data['current_status'] = $status;
        $data['statistics'] = $this->Consultation_model->get_statistics();

        $this->render('consultations/index', $data);
    }

    /**
     * Calendar view of scheduled consultations
     */
    public function calendar() {
    $this->set_title('Consultation Calendar', 'Scheduled Appointments');
    $this->add_breadcrumb('Dashboard', 'admin/dashboard');
    $this->add_breadcrumb('Consultations', 'admin/consultations');
    $this->add_breadcrumb('Calendar', null);

    // Get all scheduled consultations with admin name
    $this->db->select('consultation_requests.*, admin_users.full_name as assigned_to_name');
    $this->db->from('consultation_requests');
    $this->db->join('admin_users', 'admin_users.id = consultation_requests.assigned_to', 'left');
    $this->db->where('consultation_requests.status', 'scheduled');
    $this->db->where('consultation_requests.scheduled_date >=', date('Y-m-d'));
    $this->db->order_by('consultation_requests.scheduled_date', 'ASC');
    $query = $this->db->get();
    
    $data['scheduled_consultations'] = $query->result();

    $this->render('consultations/calendar', $data);
}
}