<?php
defined('BASEPATH') OR exit('No direct script access allowed');

// Manually load Admin_Controller if not autoloaded
if (!class_exists('Admin_Controller')) {
    require_once APPPATH . 'core/Admin_Controller.php';
}
/**
 * Admin Carousel Controller
 * Handles homepage carousel/slider management
 * Location: application/controllers/admin/Carousel.php
 */

class Carousel extends Admin_Controller {

    public function __construct() {
        parent::__construct();
    }

    /**
     * Carousel items listing
     */
    public function index() {
        $this->set_title('Home Carousel', 'Manage Slider Items');
        $this->add_breadcrumb('Dashboard', 'admin/dashboard');
        $this->add_breadcrumb('Carousel', null);

        $data['carousel_items'] = $this->Common_model->get_carousel_items();

        $this->render('carousel/index', $data);
    }

    /**
     * Add new carousel item
     */
    public function add() {
        $this->check_permission('create');
        
        $this->set_title('Add Carousel Item', 'Add New Slide');
        $this->add_breadcrumb('Dashboard', 'admin/dashboard');
        $this->add_breadcrumb('Carousel', 'admin/carousel');
        $this->add_breadcrumb('Add New', null);

        if ($this->input->method() === 'post') {
            $this->form_validation->set_rules('title', 'Title', 'required|max_length[255]');
            $this->form_validation->set_rules('media_type', 'Media Type', 'required|in_list[image,video]');

            if ($this->form_validation->run() === TRUE) {
                $data = array(
                    'title' => $this->input->post('title', TRUE),
                    'subtitle' => $this->input->post('subtitle', TRUE),
                    'description' => $this->input->post('description', TRUE),
                    'media_type' => $this->input->post('media_type', TRUE),
                    'link_url' => $this->input->post('link_url', TRUE),
                    'link_text' => $this->input->post('link_text', TRUE),
                    'display_order' => $this->input->post('display_order', TRUE) ?: 0,
                    'status' => $this->input->post('status', TRUE) ?: 'active'
                );

                // Handle media upload (image or video)
                if (!empty($_FILES['media_file']['name'])) {
                    $upload = $this->handle_upload('media_file', 'carousel', './uploads/carousel/');
                    if ($upload['status']) {
                        $data['media_path'] = $upload['data']['file_name'];
                        // Auto-detect media type if not set
                        if (isset($upload['media_type'])) {
                            $data['media_type'] = $upload['media_type'];
                        }
                    } else {
                        set_flash_message('error', $upload['error']);
                        redirect('admin/carousel/add');
                    }
                } else {
                    set_flash_message('error', 'Please upload a media file (image or video).');
                    redirect('admin/carousel/add');
                }

                if ($this->Common_model->create_carousel_item($data)) {
                    $this->log_activity('create', 'Added carousel item: ' . $data['title']);
                    set_flash_message('success', 'Carousel item added successfully!');
                    redirect('admin/carousel');
                } else {
                    set_flash_message('error', 'Failed to add carousel item. Please try again.');
                }
            }
        }

        $this->render('carousel/form');
    }

    /**
     * Edit carousel item
     */
    public function edit($id) {
        $this->check_permission('edit');
        
        $item = $this->Common_model->get_carousel_item($id);
        if (!$item) {
            show_404();
        }

        $this->set_title('Edit Carousel Item', $item->title);
        $this->add_breadcrumb('Dashboard', 'admin/dashboard');
        $this->add_breadcrumb('Carousel', 'admin/carousel');
        $this->add_breadcrumb('Edit', null);

        if ($this->input->method() === 'post') {
            $this->form_validation->set_rules('title', 'Title', 'required|max_length[255]');
            $this->form_validation->set_rules('media_type', 'Media Type', 'required|in_list[image,video]');

            if ($this->form_validation->run() === TRUE) {
              $data = array(
				'title' => $this->input->post('title', TRUE),
				'subtitle' => $this->input->post('subtitle', TRUE),
				'description' => $this->input->post('description', TRUE),
				'link_url' => $this->input->post('link_url', TRUE),
				'link_text' => $this->input->post('link_text', TRUE),
				'display_order' => $this->input->post('display_order', TRUE) ?: 0,
				'status' => $this->input->post('status', TRUE)
			);

			// Handle media upload (image or video) - ONLY if new file uploaded
			if (!empty($_FILES['media_file']['name'])) {
				// Delete old media
				if ($item->media_path) {
					$this->delete_file('./uploads/carousel/' . $item->media_path);
				}

				$upload = $this->handle_upload('media_file', 'carousel', './uploads/carousel/');
				if ($upload['status']) {
					$data['media_path'] = $upload['data']['file_name'];
					$data['media_type'] = $this->input->post('media_type', TRUE);
				} else {
					set_flash_message('error', $upload['error']);
					redirect('admin/carousel/edit/' . $id);
				}
			} 
			// If no new file uploaded, don't update media_path or media_type
			// They will remain as they were

                if ($this->Common_model->update_carousel_item($id, $data)) {
                    $this->log_activity('update', 'Updated carousel item: ' . $data['title'], $id, 'carousel');
                    set_flash_message('success', 'Carousel item updated successfully!');
                    redirect('admin/carousel');
                } else {
                    set_flash_message('error', 'Failed to update carousel item. Please try again.');
                }
            }
        }

        $data['item'] = $item;
        $this->render('carousel/form', $data);
    }

    /**
     * Delete carousel item
     */
    public function delete($id) {
        $this->check_permission('delete');
        
        $item = $this->Common_model->get_carousel_item($id);
        if (!$item) {
            set_flash_message('error', 'Carousel item not found.');
            redirect('admin/carousel');
        }

        // Delete media file
        if ($item->media_path) {
            $this->delete_file('./uploads/carousel/' . $item->media_path);
        }

        if ($this->Common_model->delete_carousel_item($id)) {
            $this->log_activity('delete', 'Deleted carousel item: ' . $item->title, $id, 'carousel');
            set_flash_message('success', 'Carousel item deleted successfully!');
        } else {
            set_flash_message('error', 'Failed to delete carousel item.');
        }

        redirect('admin/carousel');
    }

    /**
     * Update display order via AJAX
     */
    public function update_order() {
        $this->check_permission('edit');
        
        if ($this->input->method() === 'post') {
            $order_data = $this->input->post('order');
            
            if (is_array($order_data)) {
                foreach ($order_data as $order => $id) {
                    $this->Common_model->update_carousel_item($id, array('display_order' => $order));
                }
                
                $this->success_response('Display order updated successfully');
            } else {
                $this->error_response('Invalid order data');
            }
        }
    }
}