<?php
defined('BASEPATH') OR exit('No direct script access allowed');

// Manually load Admin_Controller if not autoloaded
if (!class_exists('Admin_Controller')) {
    require_once APPPATH . 'core/Admin_Controller.php';
}

/**
 * Admin Blog Controller
 * Handles blog post management (CRUD operations)
 * Location: application/controllers/admin/Blog.php
 */

class Blog extends Admin_Controller {

    public function __construct() {
        parent::__construct();
    }

    /**
     * Blog posts listing
     */
   public function index($page = 0) {
    $this->set_title('Blog Posts', 'Manage Blog Content');
    $this->add_breadcrumb('Dashboard', 'admin/dashboard');
    $this->add_breadcrumb('Blog Posts', null);

    // Pagination
    $per_page = 20;
    $status = $this->input->get('status');
    $total_rows = $this->Blog_model->count_posts($status);

    $this->load->library('pagination');
    $config = generate_pagination(base_url('admin/blog/index'), $total_rows, $per_page, 4);
    $this->pagination->initialize($config);

    $data['posts'] = $this->Blog_model->get_posts($per_page, $page, $status);
    $data['pagination'] = $this->pagination->create_links();
    $data['current_status'] = $status;

    // ADD THIS - Get statistics for all statuses
    $data['statistics'] = array(
        'total' => $this->Blog_model->count_posts(null),
        'published' => $this->Blog_model->count_posts('published'),
        'draft' => $this->Blog_model->count_posts('draft'),
        'archived' => $this->Blog_model->count_posts('archived')
    );

    $this->render('blog/index', $data);
}

    /**
     * Add new blog post
     */
    public function add() {
        $this->check_permission('create');
        
        $this->set_title('Add New Post', 'Create Blog Post');
        $this->add_breadcrumb('Dashboard', 'admin/dashboard');
        $this->add_breadcrumb('Blog Posts', 'admin/blog');
        $this->add_breadcrumb('Add New', null);

        if ($this->input->method() === 'post') {
            // Validation rules
            $this->form_validation->set_rules('title', 'Title', 'required|max_length[255]');
            $this->form_validation->set_rules('category_id', 'Category', 'required|integer');
            $this->form_validation->set_rules('content', 'Content', 'required');
            $this->form_validation->set_rules('status', 'Status', 'required|in_list[draft,published,archived]');

            if ($this->form_validation->run() === TRUE) {
                $data = array(
                    'title' => $this->input->post('title', TRUE),
                    'category_id' => $this->input->post('category_id', TRUE),
                    'author_id' => $this->admin_id,
                    'excerpt' => $this->input->post('excerpt', TRUE),
                    'content' => $this->input->post('content'),
                    'meta_title' => $this->input->post('meta_title', TRUE),
                    'meta_description' => $this->input->post('meta_description', TRUE),
                    'status' => $this->input->post('status', TRUE)
                );

                // Handle featured image upload
                if (!empty($_FILES['featured_image']['name'])) {
                    $upload = $this->handle_upload('featured_image', 'blog', './uploads/blog/');
                    if ($upload['status']) {
                        $data['featured_image'] = $upload['data']['file_name'];
                    } else {
                        set_flash_message('error', $upload['error']);
                        redirect('admin/blog/add');
                    }
                }

                if ($this->Blog_model->create_post($data)) {
                    $this->log_activity('create', 'Created blog post: ' . $data['title']);
                    set_flash_message('success', 'Blog post created successfully!');
                    redirect('admin/blog');
                } else {
                    set_flash_message('error', 'Failed to create blog post. Please try again.');
                }
            }
        }

        $data['categories'] = $this->Blog_model->get_categories();
        $this->render('blog/form', $data);
    }

    /**
     * Edit blog post
     */
    public function edit($id) {
        $this->check_permission('edit');
        
        $post = $this->Blog_model->get_post_by_id($id);
        if (!$post) {
            show_404();
        }

        $this->set_title('Edit Post', $post->title);
        $this->add_breadcrumb('Dashboard', 'admin/dashboard');
        $this->add_breadcrumb('Blog Posts', 'admin/blog');
        $this->add_breadcrumb('Edit', null);

        if ($this->input->method() === 'post') {
            $this->form_validation->set_rules('title', 'Title', 'required|max_length[255]');
            $this->form_validation->set_rules('category_id', 'Category', 'required|integer');
            $this->form_validation->set_rules('content', 'Content', 'required');
            $this->form_validation->set_rules('status', 'Status', 'required|in_list[draft,published,archived]');

            if ($this->form_validation->run() === TRUE) {
                $data = array(
                    'title' => $this->input->post('title', TRUE),
                    'category_id' => $this->input->post('category_id', TRUE),
                    'excerpt' => $this->input->post('excerpt', TRUE),
                    'content' => $this->input->post('content'),
                    'meta_title' => $this->input->post('meta_title', TRUE),
                    'meta_description' => $this->input->post('meta_description', TRUE),
                    'status' => $this->input->post('status', TRUE)
                );

                // Handle featured image upload
                if (!empty($_FILES['featured_image']['name'])) {
                    // Delete old image
                    if ($post->featured_image) {
                        $this->delete_file('./uploads/blog/' . $post->featured_image);
                    }

                    $upload = $this->handle_upload('featured_image', 'blog', './uploads/blog/');
                    if ($upload['status']) {
                        $data['featured_image'] = $upload['data']['file_name'];
                    }
                }

                if ($this->Blog_model->update_post($id, $data)) {
                    $this->log_activity('update', 'Updated blog post: ' . $data['title'], $id, 'blog_post');
                    set_flash_message('success', 'Blog post updated successfully!');
                    redirect('admin/blog');
                } else {
                    set_flash_message('error', 'Failed to update blog post. Please try again.');
                }
            }
        }
		

        $data['post'] = $post;
        $data['categories'] = $this->Blog_model->get_categories();
        $this->render('blog/form', $data);
    }

    /**
     * Archive blog post (soft delete)
     */
    public function archive($id) {
        $this->check_permission('delete');
        
        $post = $this->Blog_model->get_post_by_id($id);
        if (!$post) {
            set_flash_message('error', 'Blog post not found.');
            redirect('admin/blog');
            return;
        }

        // Update status to archived
        $data = array('status' => 'archived');
        
        if ($this->Blog_model->update_post($id, $data)) {
            $this->log_activity('archive', 'Archived blog post: ' . $post->title, $id, 'blog_post');
            set_flash_message('success', 'Blog post archived successfully!');
        } else {
            set_flash_message('error', 'Failed to archive blog post.');
        }

        redirect('admin/blog');
    }

    /**
     * Unarchive blog post
     */
    public function unarchive($id) {
        $this->check_permission('edit');
        
        $post = $this->Blog_model->get_post_by_id($id);
        if (!$post) {
            set_flash_message('error', 'Blog post not found.');
            redirect('admin/blog');
            return;
        }

        // Update status back to draft
        $data = array('status' => 'draft');
        
        if ($this->Blog_model->update_post($id, $data)) {
            $this->log_activity('unarchive', 'Unarchived blog post: ' . $post->title, $id, 'blog_post');
            set_flash_message('success', 'Blog post restored to draft!');
        } else {
            set_flash_message('error', 'Failed to restore blog post.');
        }

        redirect('admin/blog');
    }

    /**
     * Delete blog post (permanent delete)
     */
    public function delete($id) {
        $this->check_permission('delete');
        
        $post = $this->Blog_model->get_post_by_id($id);
        if (!$post) {
            set_flash_message('error', 'Blog post not found.');
            redirect('admin/blog');
        }

        // Delete featured image
        if ($post->featured_image) {
            $this->delete_file('./uploads/blog/' . $post->featured_image);
        }

        if ($this->Blog_model->delete_post($id)) {
            $this->log_activity('delete', 'Deleted blog post: ' . $post->title, $id, 'blog_post');
            set_flash_message('success', 'Blog post deleted successfully!');
        } else {
            set_flash_message('error', 'Failed to delete blog post.');
        }

        redirect('admin/blog');
    }

    /**
     * Blog categories management
     */
/**
 * Blog categories management
 */
public function categories() {
    $this->set_title('Blog Categories', 'Manage Categories');
    $this->add_breadcrumb('Dashboard', 'admin/dashboard');
    $this->add_breadcrumb('Blog Posts', 'admin/blog');
    $this->add_breadcrumb('Categories', null);

    if ($this->input->method() === 'post') {
        $action = $this->input->post('action');

        if ($action === 'add') {
            $this->form_validation->set_rules('name', 'Category Name', 'required|max_length[100]');
            
            if ($this->form_validation->run() === TRUE) {
                $data = array(
                    'name' => $this->input->post('name', TRUE),
                    'description' => $this->input->post('description', TRUE),
                    'status' => 'active'
                );

                if ($this->Blog_model->create_category($data)) {
                    $this->log_activity('create', 'Created blog category: ' . $data['name']);
                    set_flash_message('success', 'Category created successfully!');
                } else {
                    set_flash_message('error', 'Failed to create category.');
                }
            } else {
                set_flash_message('error', validation_errors());
            }
        } 
        elseif ($action === 'edit') {
            $category_id = $this->input->post('category_id');
            $this->form_validation->set_rules('name', 'Category Name', 'required|max_length[100]');
            
            if ($this->form_validation->run() === TRUE) {
                $data = array(
                    'name' => $this->input->post('name', TRUE),
                    'description' => $this->input->post('description', TRUE),
                    'status' => $this->input->post('status', TRUE)
                );

                if ($this->Blog_model->update_category($category_id, $data)) {
                    $this->log_activity('update', 'Updated blog category: ' . $data['name'], $category_id, 'blog_category');
                    set_flash_message('success', 'Category updated successfully!');
                } else {
                    set_flash_message('error', 'Failed to update category.');
                }
            } else {
                set_flash_message('error', validation_errors());
            }
        } 
        elseif ($action === 'delete') {
            $category_id = $this->input->post('category_id');
            $category = $this->Blog_model->get_category_by_id($category_id);
            
            if ($this->Blog_model->delete_category($category_id)) {
                $this->log_activity('delete', 'Deleted blog category: ' . $category->name, $category_id, 'blog_category');
                set_flash_message('success', 'Category deleted successfully!');
            } else {
                set_flash_message('error', 'Failed to delete category. Make sure no posts are using it.');
            }
        }

        redirect('admin/blog/categories');
    }

    $data['categories'] = $this->Blog_model->get_categories_with_count();
    $this->render('blog/categories', $data);
}
	
}