<?php
defined('BASEPATH') OR exit('No direct script access allowed');


// Manually load Admin_Controller if not autoloaded
if (!class_exists('Admin_Controller')) {
    require_once APPPATH . 'core/Admin_Controller.php';
}
/**
 * Admin Auth Controller
 * Handles admin login, logout, forgot password
 * Location: application/controllers/admin/Auth.php
 */

class Auth extends CI_Controller {

    public function __construct() {
        parent::__construct();
        $this->load->library('auth_library');
        $this->load->helper('MY_utility');
    }

    /**
     * Login page
     */
    public function login() {
        // Redirect if already logged in
        if ($this->auth_library->is_logged_in()) {
            redirect('admin/dashboard');
        }

        $data['page_title'] = 'Admin Login';

        // Check if form is submitted
        if ($this->input->method() === 'post') {
            // Validate form
            $this->form_validation->set_rules('username', 'Username/Email', 'required');
            $this->form_validation->set_rules('password', 'Password', 'required');

            if ($this->form_validation->run() === TRUE) {
                $username = $this->input->post('username', TRUE);
                $password = $this->input->post('password', TRUE);

                // Attempt login
                $result = $this->auth_library->login($username, $password);

                if ($result['status']) {
                    // Check for redirect URL
                    $redirect = $this->session->userdata('login_redirect');
                    if ($redirect) {
                        $this->session->unset_userdata('login_redirect');
                        redirect($redirect);
                    }
                    
                    redirect('admin/dashboard');
                } else {
                    set_flash_message('error', $result['message']);
                }
            }
        }

        // Load login view (without admin layout)
        $this->load->view('admin/auth/login', $data);
    }

    /**
     * Logout
     */
    public function logout() {
        $this->auth_library->logout();
        set_flash_message('success', 'You have been logged out successfully.');
        redirect('admin/login');
    }

    /**
     * Forgot password
     */
    public function forgot_password() {
        // Redirect if already logged in
        if ($this->auth_library->is_logged_in()) {
            redirect('admin/dashboard');
        }

        $data['page_title'] = 'Forgot Password';

        // Check if form is submitted
        if ($this->input->method() === 'post') {
            $this->form_validation->set_rules('email', 'Email', 'required|valid_email');

            if ($this->form_validation->run() === TRUE) {
                $email = $this->input->post('email', TRUE);
                
                // Generate reset token
                $result = $this->auth_library->generate_reset_token($email);
                
                if ($result['status']) {
                    // In production, send email with reset link
                    // For now, show token in flash message (for testing)
                    $reset_link = base_url('admin/reset-password/' . $result['admin_id'] . '/' . $result['token']);
                    
                    // Send password reset email
                    $this->load->helper('MY_email');
                    $subject = 'Password Reset Request';
                    $message = "
                    <h3>Password Reset Request</h3>
                    <p>You requested a password reset. Click the link below to reset your password:</p>
                    <p><a href='$reset_link'>Reset Password</a></p>
                    <p>This link will expire in 1 hour.</p>
                    <p>If you didn't request this, please ignore this email.</p>
                    ";
                    send_email($email, $subject, $message);
                    
                    set_flash_message('success', 'Password reset instructions have been sent to your email.');
                    redirect('admin/login');
                } else {
                    set_flash_message('error', $result['message']);
                }
            }
        }

        // Load forgot password view
        $this->load->view('admin/auth/forgot_password', $data);
    }

    /**
     * Reset password
     */
    public function reset_password($admin_id = null, $token = null) {
        // Redirect if already logged in
        if ($this->auth_library->is_logged_in()) {
            redirect('admin/dashboard');
        }

        if (!$admin_id || !$token) {
            show_404();
        }

        // Verify token
        if (!$this->auth_library->verify_reset_token($admin_id, $token)) {
            set_flash_message('error', 'Invalid or expired reset token. Please request a new one.');
            redirect('admin/forgot-password');
        }

        $data['page_title'] = 'Reset Password';
        $data['admin_id'] = $admin_id;
        $data['token'] = $token;

        // Check if form is submitted
        if ($this->input->method() === 'post') {
            $this->form_validation->set_rules('password', 'New Password', 'required|min_length[6]');
            $this->form_validation->set_rules('confirm_password', 'Confirm Password', 'required|matches[password]');

            if ($this->form_validation->run() === TRUE) {
                $password = $this->input->post('password', TRUE);
                
                // Reset password
                $result = $this->auth_library->reset_password($admin_id, $token, $password);
                
                if ($result['status']) {
                    set_flash_message('success', 'Password reset successfully. You can now login with your new password.');
                    redirect('admin/login');
                } else {
                    set_flash_message('error', $result['message']);
                }
            }
        }

        // Load reset password view
        $this->load->view('admin/auth/reset_password', $data);
    }
}