<?php
defined('BASEPATH') OR exit('No direct script access allowed');

// Manually load Frontend_Controller if not autoloaded
if (!class_exists('Frontend_Controller')) {
    require_once APPPATH . 'core/Frontend_Controller.php';
}

class Adult_registration extends Frontend_Controller {
    public function __construct() {
        parent::__construct();
        $this->load->model('Adult_registration_model');
        $this->load->helper('form');
        $this->load->library('form_validation');
    }

    public function index() {
        // Set meta tags
        $this->set_meta(
            'Adult French Learner Registration', 
            'Register for French language classes for adults', 
            'french learning, adult registration, language course'
        );

        // Set page heading
        $this->set_heading('French Language Registration', 'Adult Learner Form');

        // Add breadcrumbs
        $this->add_breadcrumb('Home', base_url());
        $this->add_breadcrumb('Adult French Registration', null);

        // Prepare data for the view
        $data = [
            'title' => 'Adult French Learner Registration'
        ];

        // Render the view using the custom render method
        $this->render('french_registration/adult_form', $data);
    }

    public function submit() {
        // Set validation rules
        $this->form_validation->set_rules('first_name', 'First Name', 'required|trim|max_length[100]');
        $this->form_validation->set_rules('surname', 'Surname', 'required|trim|max_length[100]');
        $this->form_validation->set_rules('age', 'Age', 'required|numeric|greater_than[0]|less_than[120]');
        $this->form_validation->set_rules('gender', 'Gender', 'required|in_list[Male,Female,Other]');
        $this->form_validation->set_rules('email', 'Email', 'required|valid_email|is_unique[french_adult_learners.email]');
        $this->form_validation->set_rules('contact_number', 'Contact Number', 'required|trim');
        $this->form_validation->set_rules('home_location', 'Home Location', 'required|trim');
        $this->form_validation->set_rules('french_level', 'French Level', 'required');

        // Log the submission attempt
        log_message('info', 'Adult French Registration Submission Attempt');

        if ($this->form_validation->run() == FALSE) {
            // Log validation errors
            log_message('error', 'Validation failed: ' . validation_errors());
            
            // If validation fails, reload the form with error messages
            $this->index();
            return;
        }

        // Prepare registration data
        $registration_data = [
            'first_name' => $this->input->post('first_name'),
            'surname' => $this->input->post('surname'),
            'age' => $this->input->post('age'),
            'gender' => $this->input->post('gender'),
            'email' => $this->input->post('email'),
            'contact_number' => $this->input->post('contact_number'),
            'home_location' => $this->input->post('home_location'),
            'french_level' => $this->input->post('french_level'),
            'field_of_work' => $this->input->post('field_of_work'),
            'favorite_french_activity' => $this->input->post('favorite_french_activity'),
            'area_of_improvement' => $this->input->post('area_of_improvement'),
            'preferred_lesson_times' => $this->input->post('preferred_lesson_times'),
            'learning_difficulties' => $this->input->post('learning_difficulties')
        ];

        // Log registration data (be careful with sensitive information)
        log_message('info', 'Registration data prepared for: ' . $registration_data['email']);

        // Attempt to register
        try {
            $result = $this->Adult_registration_model->register($registration_data);

            if ($result) {
                // Registration successful
                log_message('info', 'Adult registration successful for: ' . $registration_data['email']);
                
                // Try to send confirmation email (if function exists)
                if (function_exists('send_registration_confirmation')) {
                    try {
                        send_registration_confirmation($registration_data);
                    } catch (Exception $e) {
                        log_message('error', 'Failed to send confirmation email: ' . $e->getMessage());
                    }
                }

                $this->session->set_flashdata('success', 'Your registration has been submitted successfully!');
                redirect('french-registration/adult');
            } else {
                // Registration failed
                log_message('error', 'Adult registration failed for: ' . $registration_data['email']);
                $this->session->set_flashdata('error', 'Registration failed. Please try again.');
                $this->index();
            }
        } catch (Exception $e) {
            // Log any unexpected errors
            log_message('error', 'Unexpected error in adult registration: ' . $e->getMessage());
            $this->session->set_flashdata('error', 'An unexpected error occurred. Please try again later.');
            $this->index();
        }
    }
}